/*
 * @(#)MVCList.java	1.0 96/05/01 Eric Lunt
 *
 * Copyright (C) 1996 by Eric Lunt <elunt@mcs.net>.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

package com.bdnm.mvc;

/**
 * This class is a specialization of the standard awt
 * list box.  This one's MVC.  It holds onto a model
 * which is an instance of a SelectionInList, which basically holds
 * onto two ValueModels: one for the selection(s) and one for the list.
 * When the model changes, the contents of the list and/or the
 * selection can change.  When the user selects things in
 * the list, the underlying model changes.
 *
 * In addition, it uses an ObjectPrinter to allow the
 * use of object other than strings.
 *
 * @see SelectionInList
 * @see ObjectPrinter
 *
 * @version 1.0 05/01/96
 * @author Eric Lunt
 */

import java.awt.Event;
import java.util.Observable;
import java.util.Vector;
import java.util.Enumeration;

public class MVCList extends java.awt.List
    implements java.util.Observer {

    /**
     * This represents the underlying model for the widget
     */
    protected SelectionInList model;

    /**
     * This object is responsible for turning the objects in
     * the list to strings.
     */
    protected ObjectPrinter objectPrinter;

    /**
     * Creates a new scrolling list with no visible lines, single selection,
     * and a default model.
     */
    public MVCList() {
        this(false);
    }

    /**
     * Creates a new scrolling list with no visible lines and a default model.
     * The parameter states whether multiple selections are allowed.
     * @param   multipleSelections  if true then multiple selections are allowed
     */
    public MVCList(boolean multipleSelections) {
        this(new SelectionInList(multipleSelections), multipleSelections);
    }

    /**
     * Creates a new scrolling list with an initial model and a parameter that
     * states whether multipleSelections are allowed.
     * @param   model   the model for the widget
     * @param   multipleSelections  if true then multiple selections are allowed
     */
    public MVCList(SelectionInList model, boolean multipleSelections) {
        super(model.list().size(), multipleSelections);
        setModel(model);
    }

    /**
     * Retrieve the model.
     * @return the model the widget is using
     */
    public SelectionInList getModel() {
        return model;
    }

    /**
     * Give the widget a new model.  This will
     * also update the contents of the field.
     * @param   newModel    the new model for the widget
     */
    public synchronized void setModel(SelectionInList newModel) {
        if (model != null) {
            model.listHolder().deleteObserver(this);
            model.selectionIndexHolder().deleteObserver(this);
        }
        model = newModel;
        if (model != null) {
            model.listHolder().addObserver(this);
            model.selectionIndexHolder().addObserver(this);
            update(null,null); // This means both
        }
    }

    /**
     * Retrieve the object printer.  Lazy init one if necessary.
     * @return the object printer the widget is using
     */
    public synchronized ObjectPrinter objectPrinter() {
        if (objectPrinter == null) {
            objectPrinter = new DefaultObjectPrinter();
        }
        return objectPrinter;
    }

    /**
     * Sets a new object printer.  This will also refresh the list.
     *
     * @param   newObjectPrinter    new object printer to use
     */
    public void objectPrinter(ObjectPrinter newObjectPrinter) {
        objectPrinter = newObjectPrinter;
        refreshList();
    }

    /**
     * Before we die, remove dependency on model.
     */
    public synchronized void removeNotify() {
        if (model != null) {
            model.listHolder().deleteObserver(this);
            model.selectionIndexHolder().deleteObserver(this);
    	}
	    super.removeNotify();
    }

    /**
     * Called when observers in the observable list need to be
     * updated.  We have two observables: the selection index
     * holder and the list holder.
     *
     * @param o the observer sending the update
     * @param arg the new value
     */
    public void update(Observable o, Object arg) {
        if (o == getModel().selectionIndexHolder()) {
            updateSelections();
        } else if (o == getModel().listHolder() || o == null) {
            updateList();
        }
    }

    /**
     * Update the selections on the widget to match the
     * selections in the model.
     */
    public synchronized void updateSelections() {
        if (allowsMultipleSelections()) {
            int sel[] = getSelectedIndexes();
	        for (int i = 0 ; i < sel.length ; i++) {
	            deselect(sel[i]);
	        }
	        for (Enumeration e = getModel().selectionIndexes().elements() ; e.hasMoreElements() ;) {
	            select(((Number) e.nextElement()).intValue());
	        }
	    } else {
	        select(getModel().selectionIndex());
	    }
	}

	/**
	 * Update the list behind the widget to match the list
	 * in the model.  Make no attempt to keep the selections.
	 */
	public synchronized void updateList() {
        int sel[] = getSelectedIndexes();
	    for (int i = 0 ; i < sel.length ; i++) {
	        deselect(sel[i]);
	    }
	    delItems(0,countItems()-1);
	    for (Enumeration e = getModel().list().elements() ; e.hasMoreElements() ;) {
	        addItem(objectPrinter().toString(e.nextElement()));
	    }
	}

    /**
     * Refresh the list's elements, whose print representation may
     * have changed.  I assume that I have the same number of elements
     * in my model as in my view.
     */
    public void refreshList() {
        Vector currentList = getModel().list();
        for (int i = currentList.size() - 1; i >= 0; i--) {
            replaceItem(objectPrinter().toString(currentList.elementAt(i)), i);
        }
    }

    /**
     * We need to trap the LIST_SELECT and LIST_DESELECT events
     * so we can update the underlying model.
     *
     * Right now I only know how to deal with select for
     * single select lists.
     */
    public boolean handleEvent(Event evt) {
        if (evt.id == Event.LIST_SELECT) {
            synchronized(this) {
                getModel().selectionIndexHolder().deleteObserver(this);
                getModel().selectionIndex(((Number) evt.arg).intValue());
                getModel().selectionIndexHolder().addObserver(this);
            }
        } else if (evt.id == Event.LIST_DESELECT) {
            System.out.println(evt);
        }
        return false;
    }
}

