/*
 * @(#)MVCTextField.java	1.0 96/05/01 Eric Lunt
 *
 * Copyright (C) 1996 by Eric Lunt <elunt@mcs.net>.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

package com.bdnm.mvc;

/**
 * This class is a specialization of the standard awt
 * text field.  This one's MVC.  It holds onto a model
 * which is an instance of a ValueModel.  When the
 * model changes, the contents of the field change.
 * When the contents of the field change (on ACTION_EVENT),
 * then the widget updates the model.
 *
 * In addition, it uses a print converter to allow the
 * use of object other than strings.
 *
 * @see ValueModel
 * @see PrintConverter
 *
 * @version 1.0 05/01/96
 * @author Eric Lunt
 */

import java.awt.Event;
import java.util.Observable;

public class MVCTextField extends java.awt.TextField
    implements java.util.Observer {

    /**
     * Constant for making a String text field (default).
     */
    public final static int TYPE_STRING = 0;

    /**
     * Constant for making a Float text field.
     */
    public final static int TYPE_FLOAT = 1;

    /**
     * Constant for making a Double text field.
     */
    public final static int TYPE_DOUBLE = 2;

    /**
     * Constant for making an Integer text field.
     */
    public final static int TYPE_INTEGER = 3;

    /**
     * Constant for making a Long text field.
     */
    public final static int TYPE_LONG = 4;

    /**
     * This holds onto the model for the widget.
     */
    protected ValueModel model;

    /**
     * This holds onto the print converter for
     * this widget.
     */
    protected PrintConverter converter;

    /**
     * Default constructor
     */
    public MVCTextField() {
        super();
    }

    /**
     * Constructs a new TextField initialized with the specified columns.
     * @param cols the number of columns
     */
    public MVCTextField(int cols) {
        super(cols);
    }

    /**
     * Retrieve the model, lazy initialize one if
     * necessary.
     * @return the model the widget is using
     */
    public synchronized ValueModel getModel() {
        if (model == null) {
            setModel(new ValueHolder(getValue()));
        }
        return model;
    }

    /**
     * Return the appropriate object which the current
     * widget contents represent.  In general,
     * consumers should use ValueModel#getValue
     * on my model.
     * @return the object representing the value
     */
    protected Object getValue() {
        return (converter == null) ?
            getText() :
            converter.fromString(getText());
    }

    /**
     * Give the widget a new model.  This will
     * also update the contents of the field.
     * @param   newModel    the new model for the widget
     */
    public synchronized void setModel(ValueModel newModel) {
        if (model != null) {
            model.deleteObserver(this);
        }
        model = newModel;
        if (model != null) {
            model.addObserver(this);
            update(model,model.getValue());
        }
    }

    /**
     * Return the current print converter we are using, or null
     * if we aren't using ones (just dealing with strings).
     * @return the current print converter in use
     */
    public PrintConverter converter() {
        return converter;
    }

    /**
     * Sets the current print converter to, or null if we just want
     * to deal with strings.
     * @param   newConverter    the new converter to use
     */
    public void converter(PrintConverter newConverter) {
        converter = newConverter;
        setValue(getModel().getValue());
    }

    /**
     * If this will be used as a read-only field, then you
     * can use an ObjectPrinter instead of a PrintConverter.
     * An exception will be thrown if you attempt to read
     * from the text field.
     */
    public void converter(ObjectPrinter aPrinter) {
        converter = new ReadOnlyPrintConverter(aPrinter);
    }


    /**
     * Called when observers in the observable list need to be
     * updated.
     * @param o the observer sending the update
     * @param arg the new value
     */
    public void update(Observable o, Object arg) {
        setValue(arg);
    }

    /**
     * Set the contents of my field using the
     * given parameter.
     * @param newValue  the value that needs to be
     *                  converted to a string
     */
    private synchronized void setValue(Object newValue) {
        String newString = (converter == null) ?
                                (String) newValue :
                                converter.toString(newValue);
        if (!newString.equals(getText())) {
            setText(newString);
        }
    }

    /**
     * When the action occurs (the user accepts the
     * value shown), then make sure to update the
     * model.
     * @return whether the event was handled
     */
    public boolean action(Event event, Object what) {
        acceptValue();
       	return super.action(event,what);
    }

    /**
     * It would be great if this actually worked, but at least
     * in Win32 I never get this event.  Bummer.
     * @return whether the event was handled
     */
    public boolean lostFocus(Event event, Object what) {
       acceptValue();
       return super.lostFocus(event,what);
    }

    /**
     * Set the value of our model to the current value I represent.
     */
    private synchronized void acceptValue() {
        getModel().setValue(getValue());
    }

    /**
     * Set the type of the text field.  Right now,
     * this is a "write only" value, because we
     * don't hold onto the state.
     * @param newType   an int representing the type
     */
    public void setType(int newType) {
        switch (newType) {
            case TYPE_STRING :
                converter = null;
                break;
            case TYPE_FLOAT :
                converter = new FloatPrintConverter();
                break;
            case TYPE_DOUBLE :
                converter = new DoublePrintConverter();
                break;
            case TYPE_INTEGER :
                converter = new IntegerPrintConverter();
                break;
            case TYPE_LONG :
                converter = new LongPrintConverter();
                break;
            default :
                throw new RuntimeException("Invalid MVCTextField type: " + newType);
        }
    }

    /**
     * Before we die, remove dependency on model.
     */
    public synchronized void removeNotify() {
        if (model != null) {
            model.deleteObserver(this);
        }
        super.removeNotify();
    }

}

/**
 * This is a private PrintConverter that we'll use to
 * wrap an ObjectPrinter for read-only text fields.
 */
 class ReadOnlyPrintConverter
    implements PrintConverter {

    /**
     * This holds onto our ObjectPrinter
     */
    private ObjectPrinter objectPrinter;

    /**
     * I must create this with an ObjectPrinter.
     */
    ReadOnlyPrintConverter(ObjectPrinter anObjectPrinter) {
        super();
        objectPrinter = anObjectPrinter;
    }

    /**
     * Converts the given object to a string.  Just use
     * the ObjectPrinter we have a reference to.
     * @param   obj the object to convert to a String
     * @return a string representation of the object
     */
    public String toString(Object obj) {
        return objectPrinter.toString(obj);
    }

    /**
     * Trying to execute this method will result in a
     * run-time exception being thrown.
     * @param   stringRep   the string to convert to
     *                      an object
     * @return the converted object
     */
    public Object fromString(String stringRep) {
        throw new RuntimeException("Field must be read-only to support ObjectPrinter.  Use a PrintConverter instead.");
    }

}