/*
 * @(#)NumberValueHolder.java	1.0 96/05/01 Eric Lunt
 *
 * Copyright (C) 1996 by Eric Lunt <elunt@mcs.net>.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

package com.bdnm.mvc;

/**
 * This class represents a holder on a number.
 * It implements the Valuable interface, so you
 * can set and get this value.  In addition, it
 * is an Observable so you can share this instance
 * between any number of Observers.  The argument
 * that is sent in an update is the new value, as
 * a Number.
 *
 * This class is just like ValueHolder expect it
 * defines a number of convenience methods for
 * dealing with numerics (non-objects).  It can
 * handle ints, longs, doubles, and floats.
 *
 * @see Number
 *
 * @version 1.0 05/01/96
 * @author Eric Lunt
 */
public class NumberValueHolder extends ValueHolder {

    /**
     * Construct an instance with an initial Number.
     * @param   initValue   the initial value
     */
    public NumberValueHolder(Number initValue) {
        super(initValue);
    }

    /**
     * Construct an instance with an initial int.
     * @param   initValue   the initial value
     */
    public NumberValueHolder(int initValue) {
        this(new Integer(initValue));
    }

    /**
     * Construct an instance with an initial long.
     * @param   initValue   the initial value
     */
    public NumberValueHolder(long initValue) {
        this(new Long(initValue));
    }

    /**
     * Construct an instance with an initial float.
     * @param   initValue   the initial value
     */
    public NumberValueHolder(float initValue) {
        this(new Float(initValue));
    }

    /**
     * Construct an instance with an initial double.
     * @param   initValue   the initial value
     */
    public NumberValueHolder(double initValue) {
        this(new Double(initValue));
    }

    /**
     * Get the value as an instance of Number.
     * @return the current value, which is cast to a Number
     */
    public Number getNumberValue() {
        return (Number) getValue();
    }

    /**
     * Get the value as an int.
     * @return the current value as an int
     */
    public int intValue() {
        return getNumberValue().intValue();
    }

    /**
     * Get the value as an long.
     * @return the current value as a long
     */
    public long longValue() {
        return getNumberValue().longValue();
    }

    /**
     * Get the value as an float.
     * @return the current value as a float
     */
    public float floatValue() {
        return getNumberValue().floatValue();
    }

    /**
     * Get the value as a double.
     * @return the current value as a double
     */
    public double doubleValue() {
        return getNumberValue().doubleValue();
    }

    /**
     * Set the value as an int.
     * @param   newValue    the new value to set
     */
    public void setValue(int newValue) {
        setValue(new Integer(newValue));
    }

    /**
     * Set the value as a long.
     * @param   newValue    the new value to set
     */
    public void setValue(long newValue) {
        setValue(new Long(newValue));
    }

    /**
     * Set the value as a float.
     * @param   newValue    the new value to set
     */
    public void setValue(float newValue) {
        setValue(new Float(newValue));
    }

    /**
     * Set the value as a double.
     * @param   newValue    the new value to set
     */
    public void setValue(double newValue) {
        setValue(new Double(newValue));
    }

}