/*
 * @(#)SelectionHolder.java	1.0 96/05/01 Eric Lunt
 *
 * Copyright (C) 1996 by Eric Lunt <elunt@mcs.net>.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

package com.bdnm.mvc;

/**
 * This value model adapts to a SelectionInList instance.  It
 * appears to always have the value of the current selection.
 * This currently only works with single selection lists.
 *
 * @see SelectionInList#selectionHolder
 * @version 1.0 05/01/96
 * @author Eric Lunt
 */

import java.util.Observable;
import java.util.Observer;

public class SelectionHolder extends ValueModel
    implements Observer {

    /**
     * This is the list model we are observing and reacting to.
     */
    protected SelectionInList listModel;

    /**
     * This is how you create one of these.  Usually done
     * through SelectionInList#selectionHolder, but a direct
     * call will work too.
     * @param   aListModel  the model I am reflecting
     */
    public SelectionHolder(SelectionInList aListModel) {
        super();
        listModel = aListModel;
    }

    /**
     * Called to get the current value this object represents.
     * Necessary to implement Valuable.
     * @return the current value, which is cast to an Object.
     *          this will be either a single object or null
     */
    public Object getValue() {
        return listModel.selection();
    }

    /**
     * Called to set the current value, which must
     * be cast to an Object.  Necessary to implement
     * Valuable.
     * @param   newValue    the new value to set, which should
     *                      be either a single object or null.
     */
    public void setValue(Object newValue) {
        listModel.selection(newValue);
    }

    /**
     * The reference model changed, so propagate the update.
     * @param o should be the observable that sent the update
     * @param arg the argument being notified
     */
    public void update(Observable o, Object arg) {
        setChanged();
        notifyObservers(getValue());
    }

    /**
     * Adds an observer to the observer list.
     * @param o the observer to be added
     */
    public synchronized void addObserver(Observer o) {
        if (countObservers() == 0) {
            listModel.listHolder().addObserver(this);
            listModel.selectionIndexHolder().addObserver(this);
        }
        super.addObserver(o);
    }

    /**
     * Deletes an observer from the observer list.
     * @param o the observer to be deleted
     */
    public synchronized void deleteObserver(Observer o) {
        super.deleteObserver(o);
        if (countObservers() == 0) {
            listModel.listHolder().deleteObserver(this);
            listModel.selectionIndexHolder().deleteObserver(this);
        }
    }

    /**
     * Deletes observers from the observer list.
     */
    public synchronized void deleteObservers() {
        if (countObservers() > 0) {
            listModel.listHolder().deleteObserver(this);
            listModel.selectionIndexHolder().deleteObserver(this);
        }
        super.deleteObservers();
    }

}

