/*
 * @(#)SelectionInList.java	1.0 96/05/01 Eric Lunt
 *
 * Copyright (C) 1996 by Eric Lunt <elunt@mcs.net>.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

package com.bdnm.mvc;

/**
 * This class encapsulates two pieces of information
 * that logically sits behind a List box: the list
 * itself and the selection.  This can be used by
 * either single or multi-select list boxes.
 *
 * @see MVCList
 * @version 1.0 05/01/96
 * @author Eric Lunt
 */

import java.util.Vector;
import java.util.Enumeration;
import java.util.Observable;

public class SelectionInList
    implements java.util.Observer {

    /**
     * Constant for no selection (for single selection interface)
     */
    public final static int NO_SELECTION = -1;

    /**
     * A ValueModel on a Vector which represents the
     * list.
     */
    protected ValueModel listHolder;

    /**
     * A ValueModel on an Integer or Vector which represents the
     * selection or selections.
     */
    protected ValueModel selectionIndexHolder;

    /**
     * Create an instance with the typical holders.  This will
     * be a single select default model.
     */
    public SelectionInList() {
        this(false);
    }

    /**
     * Create an instance with the typical holders, depending
     * upon the select mode.
     * @param   isMulti true if multiple selections are allowed
     */
    public SelectionInList(boolean isMulti) {
        this(new ValueHolder(new Vector()),new ValueHolder(isMulti ? new Vector() : null));
    }

    /**
     * Create an instance with provided holders.  You
     * have to be careful with this.  Make sure that the
     * list holder is dealing with Vectors and the
     * selection index holder is dealing with Integers or
     * a Vector of Integers, depending upon the selection mode.
     * Most of the time, use the default constructor.
     * @param   listHolder  a ValueModel on a Vector
     * @param   selectionIndexHolder    a ValueModel on an Integer or a
     *                                  Vector of Integers
     */
    public SelectionInList(ValueModel listHolder, ValueModel selectionIndexHolder) {
        this.listHolder = listHolder;
        this.selectionIndexHolder = selectionIndexHolder;
        listHolder.addObserver(this);
    }

    /**
     * A ValueModel on the selection index.
     * @return the ValueModel on the selection index
     */
    public ValueModel selectionIndexHolder() {
        return selectionIndexHolder;
    }

    /**
     * A ValueModel on the list
     * @return the ValueModel on the list
     */
    public ValueModel listHolder() {
        return listHolder;
    }

    /**
     * Get the current selection index, an int.
     * This method should only be used with single
     * selection lists.
     * @return the current selection index, or
     *          -1 if there is no selection.
     */
    public int selectionIndex() {
        Object holderValue = selectionIndexHolder().getValue();
        return (holderValue == null)
            ? NO_SELECTION : ((Number) holderValue).intValue();
    }

    /**
     * Get the current selection indexes, a Vector of
     * Integers.  This method should only be used with
     * multiple selection lists.
     * @return the current selection indexes
     */
    public Vector selectionIndexes() {
        return (Vector) (selectionIndexHolder().getValue());
    }

    /**
     * Set the current selection index, an int.  You
     * can also set NO_SELECTION.  This method should
     * only be used with single selection lists.
     * @param   newIndex    an int or NO_SELECTION
     */
    public void selectionIndex(int newIndex) {
        selectionIndexHolder().setValue((newIndex == NO_SELECTION) ?
            null : new Integer(newIndex));
    }

    /**
     * Set the current selection index, an int.  You
     * can also set to null for no selections.  This method
     * should only be used with multiple selection lists.
     * @param   newIndexes  a Vector of Integers or null
     */
    public void selectionIndexes(Vector newIndexes) {
        selectionIndexHolder().setValue((newIndexes == null) ?
            new Vector() : newIndexes);
    }

    /**
     * Get the current selected object, or null if there
     * is no selection.  This method should only be used
     * with single selection lists.
     * @return the currently selected object
     */
    public Object selection() {
        int currentIndex = selectionIndex();
        Object currentSelection;
        if (currentIndex == NO_SELECTION) {
            currentSelection = null;
        } else {
            try {
                currentSelection = list().elementAt(currentIndex);
            } catch (ArrayIndexOutOfBoundsException e) {
                currentSelection = null;
            }
        }
        return currentSelection;
    }

    /**
     * Get the current selected objects.  This will
     * always return a Vector, even if empty.  This
     * method should only be used with multiple selection lists.
     * @return the currently selected objects
     */
    public Vector selections() {
        Vector currentList = list();
        Vector currentSelectionIndexes = selectionIndexes();
        Vector currentSelections = new Vector(currentSelectionIndexes.size());
        for (Enumeration e = currentSelectionIndexes.elements() ; e.hasMoreElements() ;) {
            try {
                currentSelections.addElement(currentList.elementAt(((Number) e.nextElement()).intValue()));
            } catch (ArrayIndexOutOfBoundsException exp) {
            }
        }
        return currentSelections;
    }

    /**
     * Set the current selection by providing an
     * object.  Note that I will use the #equals
     * method to determine equality.  If you pass
     * null, it clears all selections.  This method
     * should only be used with single selection lists.
     * @param   newSelection    the new selection to choose
     */
    public void selection(Object newSelection) {
        selectionIndex((newSelection == null) ?
            NO_SELECTION : list().indexOf(newSelection));
    }

    /**
     * Set the current selections by providing a Vector
     * of objects.  Note that I will use the #equals
     * method to determine equality.  If you pass
     * null, it clears all selections.  This method should
     * only be used with multiple selection lists.
     * @param   newSelection    the new selections to choose
     */
    public void selections(Vector newSelections) {
        if (newSelections == null) {
            selectionIndexes(null);
        } else {
            Vector currentList = list();
            Vector currentSelectionIndexes = new Vector(newSelections.size());
            int index;
            for (Enumeration e = newSelections.elements() ; e.hasMoreElements() ;) {
                if ((index = currentList.indexOf(e.nextElement())) >= 0) {
                    currentSelectionIndexes.addElement(new Integer(index));
                }
            }
            selectionIndexes(currentSelectionIndexes);
        }
    }

    /**
     * This is a special ValueModel that will always holds onto
     * the current selection.  It will be either an Object or null, since
     * this method only works with single-selection lists.
     * @return a selection holder
     */
    public SelectionHolder selectionHolder() {
        return new SelectionHolder(this);
    }

    /**
     * Get the current list, which is a Vector.
     * @return the current list
     */
    public Vector list() {
        return (Vector) (listHolder.getValue());
    }

    /**
     * Set the current list, which should be a Vector.  This
     * has the side effect of deselecting the current selection(s).
     * @param   newList the new list to set
     */
     public void list(Vector newList) {
        listHolder.setValue((newList == null) ? new Vector() : newList);
     }

    /**
     * Called when observers in the observable list need to be
     * updated.  We have just one observable: the listHolder.
     * When that changes, we need to make sure we update the
     * selections.
     * @param o the observer sending the update
     * @param arg the new value
     */
    public void update(Observable o, Object arg) {
        if (selectionIndexHolder().getValue() instanceof Vector) {
            selections(new Vector());
        } else {
            selection(null);
        }
    }

}
