/*
 * @(#)ValueAdaptor.java	1.0 96/05/01 Eric Lunt
 *
 * Copyright (C) 1996 by Eric Lunt <elunt@mcs.net>.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

package com.bdnm.mvc;

/**
 * This class is a ValueModel that "adapts" another
 * ValueModel, to allow for chaining models together.
 * Subclasses are responsible for implementing the
 * valueFromModel() and valueToModel() methods which
 * will convert the value as it "flows" through this
 * adaptor.
 *
 * @version 1.0 05/01/96
 * @author Eric Lunt
 */

import java.util.Observable;
import java.util.Observer;

public abstract class ValueAdaptor extends ValueModel
    implements Observer {

    /**
     * Holds onto the reference model.
     */
    private ValueModel referenceModel;

    /**
     * Construct an instance with a reference model.
     * @param   aModel   a reference model
     */
    public ValueAdaptor(ValueModel aModel) {
        super();
        setModel(aModel);
    }

    /**
     * Return the reference model that we are adapting
     * on.  Nothing special.
     * @return the reference model
     */
    public ValueModel getModel() {
        return referenceModel;
    }

    /**
     * Set the reference model.  Make sure we establish
     * the appropriate dependencies, if appropriate.
     * @param newModel  the new model to set
     */
    public synchronized void setModel(ValueModel newModel) {
        boolean haveObservers = (countObservers() > 0);
        if (haveObservers) {
            referenceModel.deleteObserver(this);
        }
        referenceModel = newModel;
        if (haveObservers && (newModel != null)) {
            newModel.addObserver(this);
            update(newModel,newModel.getValue());
        }
    }

    /**
     * The reference model changed, so propagate the
     * update.
     * @param o should be the list of observers
     * @param arg the argument being notified
     */
    public void update(Observable o, Object arg) {
        setChanged();
        notifyObservers(getValue());
    }

    /**
     * Adds an observer to the observer list.
     * @param o the observer to be added
     */
    public synchronized void addObserver(Observer o) {
        if (countObservers() == 0) {
            referenceModel.addObserver(this);
        }
        super.addObserver(o);
    }

    /**
     * Deletes an observer from the observer list.
     * @param o the observer to be deleted
     */
    public synchronized void deleteObserver(Observer o) {
        super.deleteObserver(o);
        if (countObservers() == 0) {
            referenceModel.deleteObserver(this);
        }
    }

    /**
     * Deletes observers from the observer list.
     */
    public synchronized void deleteObservers() {
        if (countObservers() > 0) {
            referenceModel.deleteObserver(this);
        }
        super.deleteObservers();
    }

    /**
     * Called to get the current value this object represents.
     * This will be my model's value modified by one of my subclass'
     * valueFromModel() method.
     * Necessary to implement Valuable.
     * @return the calculated value, which is cast to an Object
     */
    public Object getValue() {
        return valueFromModel(getModel().getValue());
    }

    /**
     * Called to set the current value, which must
     * be cast to an Object. I will set my model's value, but
     * first modifying it by one of my subclass' valueToModel()
     * method.  Necessary to implement Valuable.
     * @param   newValue    the new value to set
     */
    public void setValue(Object newValue) {
        getModel().setValue(valueToModel(newValue));
    }

    /**
     * The value coming from my model can be modified.  Return
     * the modified value that will be forwarded on.
     * @param   aValue  the value of my model
     * @return the value which I represent
     */
    protected abstract Object valueFromModel(Object aValue);

    /**
     * The value to set to my model can be modified.  Return the
     * modified value that my model will be set to.
     * @param   aValue  the value to I should represent
     * @return the value which my model should be set to
     */
     protected abstract Object valueToModel(Object aValue);
}