{******************************************************************************}
{* Project:     JavaEdit                                                      *}
{* Unit:        cEdit                                                         *}
{* Description: Child window for editing/viewing documents in Java Editor     *}
{*                                                                            *}
{* 1996, Richard L. Chase, All Rights Reserved                               *}
{*                                                                            *}
{* Types:                                                                     *}
{*    None                                                                    *}
{*                                                                            *}
{* Forms:                                                                     *}
{*    TcwEdit - Child window for editing/viewing documents. Uses a standard   *}
{*      RichEdit control. Has knowledge  of type of file being edited,        *}
{*      handles all loading, saving of document.                              *}
{*      Uses from creating form: DocTypeList, dlgSave & dlgOpen, dlgFind      *}
{*      & dlgReplace, UpdateMenuItems, DoSaveDialog, dlgPrint, StatusBar.     *}
{*                                                                            *}
{* Procedures:                                                                *}
{*    None                                                                    *}
{*                                                                            *}
{* Functions:                                                                 *}
{*    None                                                                    *}
{*                                                                            *}
{******************************************************************************}
unit cedit;

interface

uses Windows, SysUtils, Classes, Graphics, Forms, Controls, StdCtrls,
     Dialogs, Messages, Printers, Menus, Clipbrd, ComCtrls, ExtCtrls,
     javaeditdefs, Buttons, DdeMan, Search;

type
  TcwEdit = class(TForm)
    mnpText: TPopupMenu;
    mnpTextCut: TMenuItem;
    mnpTextCopy: TMenuItem;
    mnpTextPaste: TMenuItem;
    mnpTextDelete: TMenuItem;
    pnlClient: TPanel;
    StatusBar: TStatusBar;
    pnlErrors: TPanel;
    lstErrors: TListBox;
    pnlEdit: TPanel;
    Panel1: TPanel;
    cmbDocType: TComboBox;
    cmdUndo: TSpeedButton;
    cmdCut: TSpeedButton;
    cmdCopy: TSpeedButton;
    cmdPaste: TSpeedButton;
    cmdFind: TSpeedButton;
    cmdReplace: TSpeedButton;
    cmdJava: TSpeedButton;
    cmdBrowser: TSpeedButton;
    cmdPrint: TSpeedButton;
    cmdSave: TSpeedButton;
    memText: TMemo;
    mnpAddToProject: TMenuItem;
    cmdAddToProject: TSpeedButton;
    cmdRemoveFromProject: TSpeedButton;
    mnpRemoveFromProject: TMenuItem;
    cmdGoToLine: TSpeedButton;
    procedure FormClose(Sender: TObject; var Action: TCloseAction);
    procedure FormCloseQuery(Sender: TObject; var CanClose: Boolean);
    procedure doCut(Sender: TObject);
    procedure doCopy(Sender: TObject);
    procedure doPaste(Sender: TObject);
    procedure doDelete(Sender: TObject);
    procedure mnpTextPopup(Sender: TObject);
    procedure memTextClick(Sender: TObject);
    procedure memTextMouseDown(Sender: TObject; Button: TMouseButton;
      Shift: TShiftState; X, Y: Integer);
    procedure lstErrorsDblClick(Sender: TObject);
    procedure cmbDocTypeChange(Sender: TObject);
    procedure cmdUndoClick(Sender: TObject);
    procedure cmdCutClick(Sender: TObject);
    procedure cmdCopyClick(Sender: TObject);
    procedure cmdPasteClick(Sender: TObject);
    procedure cmdFindClick(Sender: TObject);
    procedure cmdReplaceClick(Sender: TObject);
    procedure cmdJavaClick(Sender: TObject);
    procedure cmdBrowserClick(Sender: TObject);
    procedure doPrint(Sender: TObject);
    procedure cmdSaveClick(Sender: TObject);
    procedure FormCreate(Sender: TObject);
    procedure memTextKeyPress(Sender: TObject; var Key: Char);
    procedure memTextKeyUp(Sender: TObject; var Key: Word;
      Shift: TShiftState);
    procedure cmdAddToProjectClick(Sender: TObject);
    procedure cmdRemoveFromProjectClick(Sender: TObject);
    procedure cmdGoToLineClick(Sender: TObject);
  private
{    FProject: TProject;}
    FDocument: TDocument;
{    FFindStart: integer;}
    FNoSaveClose: Boolean;
    FTempBrowserFile: string;
    FDDEItem: string;
    FScrollToFind: Boolean;
    function CheckFileSave: Boolean;
    function getDocType: TDocumentType;
    function getDocName: string;
    function getFileName: string;
    function getSaveStatus: boolean;
{    function getProject: TProject;}
    function CurrentLine: integer;
    procedure OpenDocument;
    procedure setDocType(DocType: TDocumentType);
    procedure setTempBrowserFile;
    procedure initDocTypes;
    procedure LoadNonNetscape(FileName: string);
    procedure NameChanged(Sender: TObject; NewName: String);
  public
    constructor CreateNew(AOwner: TComponent;
      Count: Integer; DocType: TDocumentType);
    constructor CreateOpen(AOwner: TComponent; FileName: TFileName);
    constructor CreateDoc(AOwner: TComponent; Doc: TDocument);
    function Compile: string;
    function doFind: Boolean;
    function doSave: Boolean;
    function doSaveAs: boolean;
    function HasDoc: Boolean;
    procedure doReplace;
    procedure doUndo;
    procedure GoToLine(Line: integer);
    procedure initFind;
    procedure LoadInBrowser;
    procedure UpdateToolbar;
    procedure OnDdeStart(Sender: TObject);
    procedure CloseNoSave;
    property DocumentType: TDocumentType read getDocType write setDocType;
    property DocName: string read getDocName;
    property FileName: TFileName read getFileName;
    property isSaved: Boolean read getSaveStatus;
{    property Project: TProject read getProject;} {FProject;}
  end;

implementation

{$R *.DFM}

uses fmain,
     dgotoln;

procedure TcwEdit.CloseNoSave;
begin
  FNoSaveClose := True;
  close;
end;

procedure TcwEdit.LoadNonNetscape(FileName: string);
var
  StartupInfo: TStartupInfo;
  ProcessInfo: TProcessInformation;
  CommandLine: string;
  retval: Boolean;
begin

  FillChar(StartupInfo, SizeOf(TStartupInfo), 0);

  with StartupInfo do
  begin
    cb := SizeOf(TStartupInfo);
    dwFlags := STARTF_USESHOWWINDOW or STARTF_FORCEONFEEDBACK;
    wShowWindow := SW_SHOWNORMAL;
  end;

  CommandLine := '"' + PATH_BROWSER + '" "' + FileName + '"';

  retval := CreateProcess(nil,
                       PChar(CommandLine),
                       nil,
                       nil,
                       False,
                       NORMAL_PRIORITY_CLASS,
                       nil,
                       nil,
                       StartupInfo,
                       ProcessInfo);

  if retval then begin
    with ProcessInfo do begin
      WaitForInputIdle(hProcess, INFINITE);
      CloseHandle(hThread);
      CloseHandle(hProcess);
    end;
  end else begin
    frmMain.StatusBar.SimpleText := 'Couldn''t start web browser.';
    messageBeep(0);
  end;
end;

procedure TcwEdit.setTempBrowserFile;
var
  tmpFile: string;
begin

  if (length(FTempBrowserFile) > 0) and FileExists(FTempBrowserFile) then
    {use existing file, if any}
    exit;

  tmpFile := copy(FDocument.FileName,
                  1,
                  length(FDocument.FileName) - length(ExtractFileExt(FDocument.FileName)));

  FTempBrowserFile := FDocument.FilePath + tmpFile + '.tmp';
end;

procedure TcwEdit.LoadInBrowser;
var
  DDEResult: pchar;
  LoadFileName: string;
begin

  LoadFileName := FDocument.FilePath + FDocument.FileName;

  if (not HasDoc) or (FDocument.DocType <> dtHTML) then
    exit;

  if (memText.Lines.Count = 0)
    or (length(PATH_BROWSER) = 0)
    or (not FileExists(PATH_BROWSER)) then
    exit;
{
  if (not isSaved)
    and (MessageDlg(Format('''%s' + ''' has been changed. The document must be saved before it can be loaded into your browser.',
     [FileName]),
    mtConfirmation, mbOKCancel, 0) <> mrOK) then
    exit;

  if not doSave then
    exit;
}
  if (not isSaved) then begin
    setTempBrowserFile;
    memText.Lines.SaveToFile(FTempBrowserFile);
    LoadFileName := FTempBrowserFile;
  end;

  if (pos('NETSCAPE.EXE',UpperCase(PATH_BROWSER)) = 0) then begin
    {browser not netscape}
    if (pos('APPLETVIEWER.EXE',UpperCase(PATH_BROWSER)) <> 0)
      or (pos('HOTJAVA.EXE',UpperCase(PATH_BROWSER)) <> 0) then begin
      {If browser is Appletviewer, we need to make sure the current directory
      is the one containing the file, then load with just the file name.
      for some reason, pasing a fully qualified path or URL does not work}
      ChDir(FDocument.FilePath);
      LoadNonNetscape(FDocument.FileName);
{      LoadNonNetscape('file:///' + LoadFileName)}
    end
    else LoadNonNetscape(LoadFileName);
    exit;
  end
  else if (not USEDDE) then begin
    {is netscape, but we don't want to use DDE}
    LoadNonNetscape(LoadFileName);
    exit;
  end;

  frmMain.DdeClientConv.OnOpen := OnDDEStart;

  FDDEItem := LoadFileName;
  Delete(FDDEItem,2,1);
  Insert('|',FDDEItem,2);
  FDDEItem := 'file:///' + FDDEItem + ',,'
                 + NETSCAPEWINDOW + ',' + IntToStr($1);

  if not NetscapeOpen then begin
    frmMain.DdeClientConv.ServiceApplication := PATH_BROWSER;
    frmMain.DdeClientConv.SetLink('Netscape','WWW_OpenURL');
  end
  else begin
    DDEResult := frmMain.DdeClientConv.RequestData(FDDEItem);
    try
      if DDEResult <> nil then
        NETSCAPEWINDOW := IntToStr(ord(DDEResult^));
    finally
      strDispose(DDEResult);
    end;
  end;
end;

function TcwEdit.Compile: string;
var
  tmpData: TStringList;
  FileAttribute: integer;

  procedure GoodCompile;
  begin
    StatusBar.SimpleText := Format('''%s' + ''' compiled successfully.',
     [FileName]);
  end;

  procedure BadCompile;
  var
    isError: Boolean;
    i: integer;
  begin
    if not FileExists(Result) then
      exit;
    lstErrors.Items.LoadFromFile(Result);
    isError := False;
    for i := 0 to tmpData.Count - 1 do begin
      isError := pos(' error',tmpData[i]) > 0;
      if isError then
        break;
    end;
    if  isError then begin
      StatusBar.SimpleText := 'Couldn''' + 't compile '''
                                      + FileName
                                      + '''.';
    end
    else begin
      goodCompile;
    end;
    if isError then begin
      for i := 0 to lstErrors.Items.Count - 1 do begin
        if pos(FileName + ':',lstErrors.Items[i]) > 0 then begin
          lstErrors.ItemIndex := i;
          lstErrorsDblClick(nil);
          break;
        end;
      end;
    end;
  end;

begin

  if (not HasDoc) or (FDocument.DocType <> dtJava) then
    exit;

  FileAttribute := FileGetAttr(FDocument.FilePath + FDocument.FileName);
  if (faReadOnly and FileAttribute > 0) then
    if MessageDlg(Format('%s is read only.' + chr(10) + chr(13)
      + 'The compile will not include any changes that have been made in the editor.',
      [FileName]), mtConfirmation, mbOkCancel, 0) = mrCancel then exit;

  lstErrors.Clear;
  pnlErrors.Visible := True;
  Refresh;
  Screen.Cursor := crHourglass;
  try

    StatusBar.SimpleText := 'Compiling ''' + FileName + '''...';
    application.ProcessMessages;
    try
      Result := FDocument.Compile(memText.Lines);
    finally
      StatusBar.SimpleText := '';
    end;

    if Length(Result) > 0 then begin
      if FileExists(Result) then begin
        tmpData := TStringList.Create;
        try
          tmpData.LoadFromFile(Result);
          if tmpData.Count > 0 then
            BadCompile
          else begin
            GoodCompile;
            pnlErrors.Visible := False;
          end;
        finally
          tmpData.Free;
          DeleteFile(Result);
        end;
      end;
    end
    else begin
      GoodCompile;
      pnlErrors.Visible := False;
    end;

  finally
    Screen.Cursor := crDefault;
  end;

end;

procedure TcwEdit.GoToLine(Line: integer);
var
  i, LineStart: integer;
begin
  if (Line < 0) or (Line > memText.Lines.Count) then
    exit;
  memText.SetFocus;
  LineStart := 0;
  for i := 0 to Line - 2 do
    LineStart := LineStart + length(memText.Lines[i]) + 2;
  memText.SelStart := LineStart;
  memText.SelLength := length(memText.Lines[Line-1]) + 1;
  memText.Perform( EM_SCROLLCARET, 0, 0 );
end;

function TcwEdit.CurrentLine: integer;
var
  LineEnd, i: integer;
begin
  LineEnd := 0;
  Result := 1;
  for i := 0 to memText.Lines.Count - 1 do begin
    LineEnd := LineEnd + length(memText.Lines[i]) + 2;
    if LineEnd > memText.SelStart then
      Result := i + 1;
  end;
end;
{
function TcwEdit.CurrentPos: integer;
begin
end;
}
function TcwEdit.getSaveStatus: boolean;
begin
  Result := True;
  if FDocument <> nil then
    Result := (not FDocument.isNew) and (not memText.Modified);
end;

function TcwEdit.HasDoc: boolean;
begin
  Result := FDocument <> nil;
end;

constructor TcwEdit.CreateNew(AOwner: TComponent;
  Count: Integer; DocType: TDocumentType);
begin
  inherited Create(AOwner);
  FDocument := nil;
  FScrollToFind := True;
  initDocTypes;
  memText.Color := EDITCOLOR;
  Font := frmMain.Font;
  FDocument  := TDocument.CreateNew(Count,DocType);
  FDocument.isOpen := True;
  FDocument.OnNameChange := NameChanged;
  cmbDocType.ItemIndex := ord(DocumentType);
  frmMain.UpdateFileTab(caption,FDocument.Title);
  caption := FDocument.Title;
  UpdateToolbar;
end;

constructor TcwEdit.CreateOpen(AOwner: TComponent; FileName: TFileName);
begin
  inherited Create(AOwner);
  FDocument := nil;
  FScrollToFind := True;
  initDocTypes;
  Font := frmMain.Font;
  memText.Color := EDITCOLOR;
  FDocument := TDocument.CreateOpen(FileName);
  FDocument.OnNameChange := NameChanged;
  OpenDocument;
  FDocument.IsOpen := True;
  cmbDocType.ItemIndex := ord(DocumentType)
end;

constructor TcwEdit.CreateDoc(AOwner: TComponent; Doc: TDocument);
begin
  inherited Create(AOwner);
  FDocument := nil;
  FScrollToFind := True;
  initDocTypes;
{  FProject := Doc.Project;}
  memText.Color := EDITCOLOR;
  Font := frmMain.Font;
  FDocument := Doc;
  FDocument.OnNameChange := NameChanged;
  OpenDocument;
  FDocument.isOpen := True;
  cmbDocType.ItemIndex := ord(DocumentType)
end;

procedure TcwEdit.NameChanged(Sender: TObject; NewName: String);
begin
  frmMain.UpdateFileTab(caption,NewName);
  caption := NewName;
  cmbDocType.ItemIndex := ord(DocumentType);
end;

procedure TcwEdit.OpenDocument;
begin
  Screen.Cursor := crHourglass;
  try
    frmMain.UpdateFileTab(caption,FDocument.Title);
    Caption := FDocument.Title;
    FDocument.RetrieveContents(memText.Lines);
{    memText.Lines.Assign(FDocument.Contents);}
    memText.Modified := False;
    memText.ReadOnly := ofReadOnly in frmMain.dlgOpen.Options;
    memText.SelStart := 0;
    frmMain.UpdateMenuItems(Self);
    UpdateToolbar;
  finally
    Screen.Cursor := crDefault;
  end;
end;

procedure TcwEdit.FormCloseQuery(Sender: TObject; var CanClose: Boolean);
begin
  if (not memText.Modified) or FNoSaveClose then begin
    if (length(FTempBrowserFile) > 0) and FileExists(FTempBrowserFile) then
      DeleteFile(FTempBrowserFile);
    exit;
  end;
  if FDocument.isNew then begin
    if MessageDlg(Format('Save %s?', ['''' + Caption + '''']),
        mtConfirmation, [mbYes, mbNo], 0) = idYes then
      if not doSaveAs then
        CanClose := False;
  end
  else
    CanClose := CheckFileSave;
  if CanClose then
    if (length(FTempBrowserFile) > 0) and FileExists(FTempBrowserFile) then
      DeleteFile(FTempBrowserFile);
end;

procedure TcwEdit.FormClose(Sender: TObject; var Action: TCloseAction);
begin
  frmMain.RemoveFileTab(caption);
  {if the document is assigned to a project,
   let the project handle it's destruction}
  if FDocument.Project = nil then
    FDocument.Free
  else FDocument.isOpen := False;
  Fdocument := nil;
  Action := caFree;
end;

function TcwEdit.getDocName: string;
begin
  if HasDoc then
    Result := FDocument.FileName
  else Result := '';
end;

function TcwEdit.getfileName: string;
begin
  if HasDoc then
  Result := FDocument.FilePath + FDocument.FileName;
end;

procedure TcwEdit.setDocType(DocType: TDocumentType);
begin
  FDocument.DocType := DocType;
  frmMain.UpdateFileTab(caption,FDocument.Title);
  caption := FDocument.Title;
  memText.Modified := True;
end;

function TcwEdit.getDocType: TDocumentType;
begin
  Result := dtOther;
  if HasDoc then
    Result := FDocument.DocType;
end;

function TcwEdit.CheckFileSave: Boolean;
var
  SaveResp: Integer;
begin
  result := true;
  if not memText.Modified then
    Exit;

  SaveResp := MessageDlg(Format('Save changes to %s?', [FileName]),
    mtConfirmation, mbYesNoCancel, 0);
  case SaveResp of
    idYes: doSave;
    idNo: {Nothing};
    idCancel: Result := False;
  end;
end;

procedure TcwEdit.initFind;
begin
{  FFindStart := memText.SelStart;}
end;

function TcwEdit.doSaveAs: boolean;
var
  AddToProject: Boolean;
begin
  Result := False;

  if not frmMain.DoSaveDocDialog(FDocument.DocType) then
    exit;

  if (ExtractFileExt(frmMain.dlgSave.FileName) = '') then begin
    frmMain.dlgSave.FileName := frmMain.dlgSave.FileName + '.' + ExtensionFromType(FDocument.DocType);
  end;
(* Don't need - handled by file dialog
    if FileExists(frmMain.dlgSave.FileName) then begin
      SaveResp := MessageDlg(Format('OK to overwrite %s', [frmMain.dlgSave.FileName]),
        mtConfirmation, mbYesNoCancel, 0);
      case SaveResp of
        idYes: {Nothing};
        idNo: begin
          Result := True;
          Exit;
        end;
        idCancel: Exit;
      end;
    end;
*)
  memText.Lines.SaveToFile(frmMain.dlgSave.FileName);
  {if the document is new, or if it is not currently part of the active project,
  make new doc part of the project}
  AddToProject :=  FDocument.isNew and (frmMain.Project <> nil);
{  if (FDocument.Project <> nil) and FDocument.Project.InProject(FDocument.FileName) then begin
    FDocument.Project.Remove(FDocument.FileName);
    AddToProject := True;
  end;
}
  FDocument.Load(frmMain.dlgSave.FileName);
  if AddToProject then
    frmMain.Project.Add(FDocument);
  memText.Modified := False;
  Result := True;

  frmMain.SetDocList(frmMain.dlgSave.FileName);

  frmMain.UpdateMenuItems(Self);

end;

function TcwEdit.doSave: Boolean;
var
  FileAttribute: integer;
begin
  if FDocument.isNew then
    Result := doSaveAs
  else begin
    FileAttribute := FileGetAttr(FDocument.FilePath + FDocument.FileName);
    if (faReadOnly and FileAttribute > 0) then
      MessageDlg(Format('%s is read only and will not be saved.',
      [FDocument.FilePath + FDocument.FileName]),
      mtWarning, [mbOK], 0)
    else begin
      FDocument.Save(memText.Lines);
      memText.Modified := False;
      frmMain.SetDocList(FDocument.FilePath + FDocument.FileName);
    end;
    Result := True;
  end;
  frmMain.UpdateMenuItems(Self);
end;

procedure TcwEdit.doUndo;
begin
  with memText do
    if HandleAllocated then SendMessage(Handle, EM_UNDO, 0, 0);
end;

procedure TcwEdit.doCut(Sender: TObject);
begin
  memText.CutToClipboard;
end;

procedure TcwEdit.doCopy(Sender: TObject);
begin
  memText.CopyToClipboard;
end;

procedure TcwEdit.doPaste(Sender: TObject);
begin
  memText.PasteFromClipboard;
end;

procedure TcwEdit.doDelete(Sender: TObject);
begin
  memText.ClearSelection;
end;

function TcwEdit.doFind;
var
  isFound: Boolean;
begin
 {Dick,

the behaviour of the multiline edit control has changed slightly in Win32,
you are now supposed to send a EM_SCROLLCARET message to the edit to have it
scroll the caret/selection into view, EM_SETSEL (what SelStart/SelLEngth is
using) will not do that anymore.
  SendMessage( memo1,Handle, EM_SCROLLCARET, 0, 0 );
 or
  memo1.Perform( EM_SCROLLCARET, 0, 0 );

Peter}

  memText.SetFocus;
  isFound := SearchMemo(memText,frmMain.dlgFind.FindText,frmMain.dlgFind.Options);
  if isFound then begin
    if FScrollToFind then
      memText.Perform( EM_SCROLLCARET, 0, 0 );
  end
  else begin
    frmMain.StatusBar.SimpleText := 'Can''t find ''' + frmMain.dlgFind.FindText + '''';
    messageBeep(0);
  end;

(*FIND RICHTEXT
  SearchType := [];

  if (frWholeWord in frmMain.dlgFind.Options) then
    SearchType := SearchType + [stWholeWord];
  if (frMatchCase in frmMain.dlgFind.Options) then
    SearchType := SearchType + [stMatchCase];

  doLoop := True;
  while doLoop do begin

    StartStr := memText.SelStart;
    if upperCase(memText.SelText) = UpperCase(frmMain.dlgFind.FindText) then
      StartStr := StartStr + length(frmMain.dlgFind.FindText);

    i := memText.FindText(frmMain.dlgFind.FindText, StartStr, length(memText.Text) - StartStr, SearchType);

    isFound := (i > -1);}
    if isFound Then begin
      memText.SelStart := i;
      memText.SelLength := length(frmMain.dlgFind.FindText);

      doLoop := False
    end
    else begin
      if memText.SelStart > 0 then
        memText.SelStart := 0
      else begin
        memText.SelStart := FFindStart;
        doLoop := False;
        frmMain.StatusBar.SimpleText := 'Can''t find ''' + frmMain.dlgFind.FindText + '''';
        messageBeep(0);
      end;
    end;
  end;
*)

  result := isFound;

end;

procedure TcwEdit.doReplace;
var
  StrA, StrB: string;
  initPos: integer;
begin

  if frmMain.dlgReplace = nil then
    exit;

  StrA := (memText.SelText);
  StrB := (frmMain.dlgReplace.FindText);

  if not (frMatchCase in frmMain.dlgReplace.Options) then begin
    StrA := UpperCase(StrA);
    StrB := UpperCase(StrB);
  end;

  if StrA = StrB then begin
    memText.SelText := frmMain.dlgReplace.ReplaceText;
    memText.SelLength := 0;
    memText.SelStart := memText.SelStart + length(frmMain.dlgReplace.ReplaceText);
  end;

  if (frReplaceAll in frmMain.dlgReplace.Options) then begin
    memText.SelStart := 0;
    InitPos := -1;
    FScrollToFind := False;
    try
    While doFind do begin

      {if we are replacing a string with a string that contains the string we are
      replacing, we could get caught in an infinite loop}
      {with first find, set the initial position}
      if InitPos = -1 then
        InitPos := memText.SelStart
      else
        {when SelStart is less or equal to the initial find, that
        means we've already gone through the document once.}
        if memText.SelStart <= InitPos then
          Break;

      memText.SelText := frmMain.dlgReplace.ReplaceText;
      memText.SelLength := 0;
      memText.SelStart := memText.SelStart + length(frmMain.dlgReplace.ReplaceText);
    end;
    finally
      FScrollToFind := True;
    end;
    memText.Perform( EM_SCROLLCARET, 0, 0 );
    if InitPos >= 0 then
      frmMain.StatusBar.SimpleText := 'Done Replacing '''
        + frmMain.dlgReplace.FindText + ''' with '''
        + frmMain.dlgReplace.ReplaceText + '.''';

    messageBeep(0);

  end;

  memText.Perform( EM_SCROLLCARET, 0, 0 );

end;

procedure TcwEdit.mnpTextPopup(Sender: TObject);
begin
  mnpTextPaste.Enabled := Clipboard.HasFormat(CF_TEXT);
  mnpTextCut.Enabled := memText.SelLength > 0;
  mnpTextCopy.Enabled := memText.SelLength > 0;
  mnpTextDelete.Enabled := memText.SelLength > 0;
  if (frmMain.Project <> nil) then begin
    mnpRemoveFromProject.Visible := frmMain.Project.InProject(DocName);
    mnpAddToProject.Visible := not mnpRemoveFromProject.Visible;
    mnpAddToProject.Enabled := (not frmMain.Project.InProject(DocName));
  end
  else mnpAddToProject.Enabled := False;
end;

procedure TcwEdit.memTextClick(Sender: TObject);
begin
  initFind;
  frmMain.UpdateMenuItems(Sender);
end;

procedure TcwEdit.memTextMouseDown(Sender: TObject; Button: TMouseButton;
  Shift: TShiftState; X, Y: Integer);
begin
  initFind;
end;

procedure TcwEdit.lstErrorsDblClick(Sender: TObject);
var
  LineText: string;
  BeginErrorLine, EndErrorLine: integer;
  ErrorLine: string;
begin
  LineText := lstErrors.Items[lstErrors.ItemIndex];
  beginErrorLine :=  pos(FileName + ':',LineText);

  if beginErrorLine = 0 then
    exit;

  beginErrorLine := beginErrorLine + length(FileName) + 1;
  EndErrorLine := pos(':',copy(LineText,
                               beginErrorLine,
                               length(LineText) - beginErrorLine));
  if EndErrorLine = 0 then
    exit;

  dec(EndErrorLine);
  ErrorLine := copy(LineText,BeginErrorLine,EndErrorLine);

  GoToLine(StrToInt(ErrorLine));

end;

procedure TcwEdit.cmbDocTypeChange(Sender: TObject);
begin
  DocumentType := TDocumentType(cmbDocType.ItemIndex);
end;

procedure TCWEdit.initDocTypes;
var
  i: integer;
begin
  cmbDocType.Items.clear;
  for i := 0 to frmMain.DocTypeList.Count - 1 do
    cmbDocType.Items.Add(FrmMain.DocTypeList[i]);
end;

procedure TcwEdit.cmdUndoClick(Sender: TObject);
begin
  doUndo;
end;

procedure TcwEdit.cmdCutClick(Sender: TObject);
begin
  doCut(nil);
end;

procedure TcwEdit.cmdCopyClick(Sender: TObject);
begin
  doCopy(nil);
end;

procedure TcwEdit.cmdPasteClick(Sender: TObject);
begin
  doPaste(nil);
end;

procedure TcwEdit.cmdFindClick(Sender: TObject);
begin
  initFind;
{
  pnlSearch.Visible := true;
  edtFind.SetFocus;
}
  frmMain.dlgFind.Execute;

end;

procedure TcwEdit.cmdReplaceClick(Sender: TObject);
begin
  initFind;
  frmMain.dlgReplace.Execute;
end;

procedure TcwEdit.cmdJavaClick(Sender: TObject);
begin
  if (memText.Lines.Count = 0) or (Length(PATH_JAVAC) = 0) then
    exit;
  Compile;
end;

procedure TcwEdit.UpdateToolBar;
var
  docisnew: boolean;
begin
  if FDocument = nil then
    docisnew := true
  else docisNew := FDocument.isNew;
  cmdSave.Enabled := (memText.Modified or DocisNew);
  cmdPrint.Enabled := (frmMain.dlgPrint <> nil)
                   and (memText.Lines.Count > 0)
                   and (Printer.Printers.Count > 0);
  cmdFind.Enabled := (memText.Lines.Count > 0);
  cmdReplace.Enabled := cmdFind.Enabled;
  cmdGotoLine.Enabled := cmdFind.Enabled;
  cmdJava.Visible := (DocumentType = dtJava);

  if (frmMain.Project <> nil) then begin
    cmdRemoveFromProject.Visible := frmMain.Project.InProject(DocName);
    cmdAddToProject.Visible := not cmdRemoveFromProject.Visible;
    cmdAddToProject.Enabled := (not frmMain.Project.InProject(DocName));
  end
  else cmdAddToProject.Enabled := False;

  if (DocumentType = dtJava) then
    cmdJava.Enabled := cmdFind.Enabled and (not DocisNew)
                     and (length(PATH_JAVAC) > 0)
                     and FileExists(PATH_JAVAC);
  cmdBrowser.Visible := (DocumentType = dtHtml);
  if (DocumentType = dtHtml) then
    cmdBrowser.Enabled := cmdFind.Enabled and (not DocisNew)
                       and (length(PATH_BROWSER) > 0)
                       and FileExists(PATH_BROWSER);
{  cmdCut.Enabled := memText.SelLength > 0;}
{  cmdCopy.Enabled := cmdCut.Enabled;}
  cmdPaste.Enabled := Clipboard.HasFormat(CF_TEXT);
end;

procedure TcwEdit.cmdBrowserClick(Sender: TObject);
begin
  LoadInBrowser;
end;

procedure TcwEdit.doPrint(Sender: TObject);
var
  i, copies: integer;
  PrintText: TextFile;   {declares a file variable}
begin

  if (frmMain.dlgPrint = nil) or (Printer.Printers.Count = 0)then
    exit;

  if not frmMain.dlgPrint.Execute then
    exit;

  Printer.Canvas.Font := Self.Font;  {assigns Font settings to the canvas}
  {Printer should print in Black and white, regardless of screen settings}
  Printer.Canvas.Font.Color := clBlack;
{  printer.NewPage;}
  AssignPrn(PrintText);   {assigns PrintText to the printer}
  Rewrite(PrintText);     {creates and opens the output file}
  try
    for copies := 1 to frmMain.dlgPrint.Copies do begin
      if frmMain.dlgPrint.PrintRange = prSelection then
        Writeln(PrintText,memText.SelText)
      else
        for i := 0 to memText.Lines.Count - 1 do
          Writeln(PrintText, memText.Lines[i]);	{writes the contents of the Memo1 to the printer object}
    end;
  finally
    CloseFile(PrintText); {Closes the printer variable}
  end;
{
  printer.Canvas.Font := self.Font;
  TextTop := 0;
  Printer.BeginDoc;
  try
    printer.NewPage;
    for i := 0 to memText.Lines.Count - 1 do begin
      printer.canvas.textout(0,TextTop, memText.Lines[i]);
      TextTop := TextTop + Printer.Canvas.Font.Height + 3;
    end;
  finally
    Printer.EndDoc;
  end;
}
end;

procedure TcwEdit.cmdSaveClick(Sender: TObject);
begin
  dosave;
end;

procedure TcwEdit.FormCreate(Sender: TObject);
begin
  FNoSaveClose := False;
end;

procedure TcwEdit.OnDdeStart(Sender: TObject);
var
  DDEResult: PChar;
begin
  NETSCAPEOPEN := True;
  application.processmessages;
  DDEResult := frmMain.DdeClientConv.RequestData(FDDEItem);
  try
    if DDEResult <> nil then
      NETSCAPEWINDOW := IntToStr(ord(DDEResult^));
  finally
    strDispose(DDEResult);
  end;
end;

procedure TcwEdit.memTextKeyPress(Sender: TObject; var Key: Char);
begin
  case Key of
    #9: begin
      if TABSET <> -1 then begin
        Key := #0;
        memText.SelText := StringOfChar(' ',TABSET);
      end;
    end;
  end;
end;

(*
procedure TForm1.FindDialog1Find(Sender: TObject);
var
ToFind: string;         {String to find}
FindIn: string;         {String to search}
Found: integer;         {String found indicator}
Index: integer;         {Start pos. of found text}
FoundLen: integer;      {Length of found text}
begin
ToFind := FindDialog1.FindText;     {User specified string in dialog}
FoundLen := Length(FindDialog1.FindText); {Length of specified text}
FindIn := Form1.Edit1.Text;      {Text to search for string}

Found := Pos(ToFind, FindIn);    {Search for string}
If Found > 0 then
  begin
  Form1.BringToFront;             {Edit control must have focus in
  Form1.ActiveControl := Edit1;    order to display selected text}
  Edit1.SelStart:= Found -1;      {Select the string found by POS}
  Edit1.SelLength := FoundLen;
  end;
Else
  begin
  MessageDlg('Text not found', mtInformation, [mbOK], 0);
  end;
end;
*)
procedure TcwEdit.memTextKeyUp(Sender: TObject; var Key: Word;
  Shift: TShiftState);
var
  i: integer;
  LastLine: string;
  InsertChars: string;
begin
  InsertChars := '';
  LastLine := '';
  if Key = VK_RETURN then begin
    if AUTOINDENT then begin
      if memText.Lines.Count > 0 then begin
      {go back to 0 or previous crlf}
        for i := (memtext.selstart -2) downto 1 do
          if (memText.Text[i] = #10) or (memText.Text[i] = #13) then
            break
          else LastLine := LastLine + memText.Text[i];
        {get pos of first non-space or tab char in previous line}
        for i := length(LastLine) downto 1 do
          if (LastLine[i] <> #32) and (LastLine[i] <> #9)then
            break
          else InsertChars := InsertChars + LastLine[i];{inc(Indent);}
        {add spaces accordingly}
        if length(InsertChars) > 0 then
          memText.SelText := InsertChars;
      end;
    end;
  end;
  frmMain.UpdateMenuItems(Sender);
end;

procedure TcwEdit.cmdAddToProjectClick(Sender: TObject);
begin

  if frmMain.Project = nil then
    exit;

  if FDocument.isNew then begin
    if not doSaveAs then
      exit;
  end;

  frmMain.Project.Add(FDocument);

  frmMain.UpdateMenuItems(nil);

end;

procedure TcwEdit.cmdRemoveFromProjectClick(Sender: TObject);
begin

  if frmMain.Project = nil then
    exit;

  frmMain.Project.Remove(DocName);

  frmMain.UpdateMenuItems(nil);

end;

procedure TcwEdit.cmdGoToLineClick(Sender: TObject);
var
  dlgGotoLine: TdlgGoToLine;
begin
  dlgGoToLine := TdlgGoToLine.Create(frmMain);
  try
    dlgGoToLine.ShowModal;
    if dlgGoToLine.ModalResult = mrOK then begin
      GoToLine(dlgGoToLine.Line);
    end;
  finally
    dlgGoToLine.Free;
  end;
end;

end.
