unit Essapi32connection;

interface

uses essapi32, SysUtils, WinTypes, WinProcs, Messages, Classes,
     Forms, Controls, DsgnIntf, Dialogs, EssList;

type

{Errors}
  EEssInitErr = Class(Exception);

{Forward}
  TEssSvrConnection = class;
  TEssDBConnection = class;

  TEssSession = class(TComponent)
  private
    FBeforeStart: TNotifyEvent;
    FAfterStart: TNotifyEvent;
    FInstance: ESS_HINST_T;
    FMaxHandles:   ESS_USHORT_T; (* max number of context handles required *)
    FMaxBuffer:    ESS_SIZE_T;   (* max size of buffer that can be allocated *)
    FLocalPath:    ESS_PATH_T;    (* local path to use for file operations *)
    FMessagePath:  ESS_PATH_T;
    FSvrConnections: TList;
    function getLocalPath: string;
    procedure setLocalPath(PathName: string);
    procedure RemoveConnection(Connection: TEssSvrConnection);
    function getMessagePath: TFileName;
    procedure setMessagePath(PathName: TFileName);
    function getStatus: Boolean;
    procedure setStatus(Option: Boolean);
    procedure Start;
    procedure Stop;
  protected
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    property Instance: ESS_HINST_T read FInstance;
  published
    property Started: Boolean read getStatus write setStatus;
    property MaxHandles: ESS_USHORT_T read FMaxHandles write FMaxHandles;
    property MaxBuffer: ESS_SIZE_T read FMaxBuffer write FMaxBuffer;   (* max size of buffer that can be allocated *)
    property LocalPath: string read getLocalPath write setLocalPath;    (* local path to use for file operations *)
    property MessagePath: TFileName read getMessagePath write setMessagePath;    (* local path to use for file operations *)
    Property BeforeStart: TNotifyEvent read FBeforeStart write FBeforeStart;
    Property AfterStart: TNotifyEvent read FAfterStart write FAfterStart;
  end;

{******************************************************************************}
{TEssSvrConnection -  Connection to an Essbase Server
{******************************************************************************}
  TEssSvrConnection = class(TComponent)
  private
    FServer: ESS_SVRNAME_T;
    FUserName: ESS_USERNAME_T;
    FPassword: ESS_PASSWORD_T;
    FAppList: TStringList;
    FDBList: TStringList;
    FAppDBList: TStringList;
    FContext: ESS_HCTX_T;
    FSaveLoginInfo: Boolean;
    FSavePassword: Boolean;
    FDBConnections: TList;
    function getConnected: Boolean;
    function getPassword: string;
    function getSavePassword: Boolean;
    function getServer: string;
    function getUserName: string;
    function getInstance: ESS_HINST_T;
    procedure setConnected(Value: Boolean);
    procedure setPassword(Value: string);
    procedure setSaveLoginInfo(Value: Boolean);
    procedure setSavePassword(Value: Boolean);
    procedure setServer(Value: string);
    procedure setUserName(Value: string);
  protected
    procedure connect;
    procedure disconnect;
    procedure RemoveDBConnection(DBConnection: TEssDBConnection);
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    property ContextHandle: ESS_HCTX_T read FContext;
    property Instance: ESS_HINST_T read getInstance;
  published
    property Connected: Boolean read getConnected write setConnected stored False;
    property Password: string read getPassword write setPassword stored FSavePassword;
    property SaveLoginInfo: Boolean read FSaveLoginInfo write setSaveLoginInfo;
    property SavePassword: Boolean read getSavePassword write setSavePassword;
    property Server: string read getServer write setServer stored FSaveLogInInfo;
    property UserName: string read getUserName write setUserName stored FSaveLoginInfo;
  end;

  TEssAutoLogin = (alDefault, alNoDialog, alNoSelect);

{forward}
  TEssDataSource = class;
  TEssMemberBase = class;
  TEssMember = class;
  TEssMemberList = class;
  TEssDatabase = class;

{******************************************************************************}
{TEssDBConnection -  Connection to an Essbase database
{******************************************************************************}
  TEssDBConnection = class(TComponent)
  private
    FSvrConnection: TEssSvrConnection;
    FShowDbList: Boolean;
    FConnected: Boolean;
    FAppDbIndex: integer;
    FAppName: ESS_APPNAME_T;
    FDbName: ESS_DBNAME_T;
    FAccess: ESS_ACCESS_T;
    FOptions: TEssAutoLogin;
    FBeforeConnect: TNotifyEvent;
    FAfterConnect: TNotifyEvent;
    FDatabase: TEssDatabase;
    FEssDataSources: TList;
    FSaveLoginInfo: Boolean;
{    FSavePassword: Boolean;}
    FQueryForMembers: Boolean;
    function getSvrConnection: TEssSvrConnection;
    procedure setSvrConnection(NewConnection: TEssSvrConnection);
    function getServer: string;
    function getUserName: string;
    function getPassword: string;
    function getContext: ESS_HCTX_T;
    function getAppName: string;
    function getDbName: string;
{    function getSavePassword: Boolean;}
    function getConnected: Boolean;
    function getInstance: ESS_HINST_T;
    function getInitialized: boolean;
    procedure setAppName(Value: string);
    procedure setDbName(Value: string);
    procedure setConnected(Value: Boolean);
    procedure setInitialized(Value: Boolean);
    procedure setSaveLoginInfo(Value: Boolean);
{    procedure setSavePassword(Value: Boolean);}
  protected
    procedure doConnect(Value: Boolean); virtual;
    procedure connect;
    procedure disconnect;
    procedure RemoveEssDataSource(EssDataSource: TEssDataSource);
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    function GetMemberDimension(mbrName: string): string;
    function Member(MName: string): TEssMemberBase;
    procedure GetDimensionList(Sender: TComponent; List: TStrings);
    procedure QueryMemberData(Query: pchar; Dest: TStringList);
    property ContextHandle: ESS_HCTX_T read getContext;
    property Initialized: Boolean read getInitialized write setInitialized;
    property Instance: ESS_HINST_T read getInstance;
    property Database: TEssDatabase read FDatabase;
  published
    property AppName: string read getAppName write setAppName stored FSaveLoginInfo;
    property SvrConnection: TEssSvrConnection read getSvrConnection write setSvrConnection;
    property Connected: Boolean read getConnected write setConnected stored false;
    property DbName: string read getDbName write setDBName stored FSaveLoginInfo;
    property QueryForMembers: Boolean read FQueryForMembers write FQueryForMembers;
    property SaveLoginInfo: Boolean read FSaveLoginInfo write setSaveLoginInfo;
{    property SavePassword: Boolean read getSavePassword write setSavePassword;}
    property ShowDBList: Boolean read FShowDBList write FShowDBList;
    Property BeforeConnect: TNotifyEvent read FBeforeConnect write FBeforeConnect;
    Property AfterConnect: TNotifyEvent read FAfterConnect write FAfterConnect;
  end;

{******************************************************************************}
{TEssDataSource -  communication link between connection and other objects.
{******************************************************************************}
  TEssDataSource = class(TComponent)
  private
    FOnConnect: TnotifyEvent;
    FOnDisconnect: TnotifyEvent;
    FConnection: TEssDBConnection;
    function getConnection: TEssDBConnection;
    procedure setConnection(NewConnection: TEssDBConnection);
  protected
    procedure doConnect(Value: Boolean); virtual;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    function Member(MName: string): TEssMemberBase;
    procedure GetDimensionList(Sender: TComponent; List: TStrings);
    property Connection: TEssDBConnection read getConnection write setConnection;
    property OnConnect: TNotifyEvent read FOnConnect write FOnConnect;
    property OnDisconnect: TNotifyEvent read FOnDisconnect write FOnDisconnect;
  end;

{******************************************************************************}
{TEssMemberBase - base class for members and dimensions}
{******************************************************************************}
  TEssMemberBase = class
  private
    FName: string;
    FDimName: string;
    FAlias: string;
    FOwner: TEssDBConnection;
    FMemberList: TEssMemberList;
    function getAlias: string;
  protected
  public
    constructor Create(AOwner: TEssDBConnection; ObjName: string); virtual;
    destructor Destroy; override;
    function mbrCount: integer;
    function Member(MName: string): TEssMemberBase;
    function isDescendant(MName: string): Boolean;
    function HasChildren: Boolean;
{    procedure MembersOfSetting(Setting: TEssMemberSetting; FillList: TStringList);}
    procedure AllInSameDim(FillList: TStringList);
    procedure AllSiblings(FillList: TStringList);
    procedure Ancestors(FillList: TStringList);
    procedure Children(FillList: TStringList);
    procedure Descendants(FillList: TStringList);
    procedure DimBottom(FillList: TStringList);
    procedure DimTop(FillList: TStringList);
    procedure IAncestors(FillList: TStringList);
    procedure IChildren(FillList: TStringList);
    procedure IDescendants(FillList: TStringList);
    procedure OfSameGen(FillList: TStringList);
    procedure OnSameLevel(FillList: TStringList);
    procedure Parent(FillList: TStringList);
    property Name: string read FName;
    property DimName: string read FDimName;
    property Alias: string read getAlias;
    property Members: TEssMemberList read FMemberList;
  end;

{******************************************************************************}
{TEssMember - a member of a dimension}
{******************************************************************************}
  TEssMember = class(TEssMemberBase)
  private
    FInitialized: Boolean;
    FParentMbrName: string;
    FChildMbrName: string;
    FPrevMbrName: string;
    FNextMbrName: string;
    FLevel: ESS_SHORT_T;
    FGeneration: ESS_SHORT_T;
    FDescription: string;
    procedure Initialize;
  protected
  public
    constructor Create(AOwner: TEssDBConnection; memName, memDimName: string;
      memLevel, memGeneration: ESS_SHORT_T);
    destructor Destroy; override;
    function HasChildren: Boolean;
{    function InitChildren(Refresh: Boolean): Boolean;}
    property ParentMbrName: string read FParentMbrName;
    property ChildMbrName: string read FChildMbrName;
    property PrevMbrName: string read FPrevMbrName;
    property NextMbrName: string read FNextMbrName;
    property Level: ESS_SHORT_T read FLevel;
    property Generation: ESS_SHORT_T read FGeneration;
    property Description: string read FDescription;
  end;

{******************************************************************************}
{TEssMemberList - a list of members}
{******************************************************************************}
  TEssMemberList = class(TEssList)
  private
  protected
    function Get(Index: Integer): TEssMember;
    procedure Put(Index: Integer; Item: TEssMember);
  public
    destructor Destroy; override;
    procedure Clear; override;
    procedure Add(NewMember: TEssMember);
    property Member[Index: Integer]: TEssMember read get write put; default;
  end;

{******************************************************************************}
{TEssDimension - an database dimension}
{******************************************************************************}
  TEssDimension = class(TEssMemberBase)
  private
    FDeclDimSize: ESS_ULONG_T;
    FActDimSize: ESS_ULONG_T;
    FChildMbrName: string;
    FPrevMbrName: string;
    FNextMbrName: string;
    FLevel: ESS_SHORT_T;
    FDescription: string;
  protected
  public
    constructor Create(AOwner: TEssDBConnection; DimStats: ESS_DIMSTATS_T);
    destructor Destroy; override;
    property DeclaredDimSize: ESS_ULONG_T read FDeclDimSize;
    property ActualDimSize: ESS_ULONG_T read FActDimSize;
    property ChildMbrName: string read FChildMbrName;
    property PrevMbrName: string read FPrevMbrName;
    property NextMbrName: string read FNextMbrName;
    property Level: ESS_SHORT_T read FLevel;
    property Description: string read FDescription;
  end;

{******************************************************************************}
{TEssDimensionList - a list of database dimensions}
{******************************************************************************}
  TEssDimensionList = class(TEssList)
  private
  protected
    function Get(Index: Integer): TEssDimension;
    procedure Put(Index: Integer; Item: TEssDimension);
  public
    destructor Destroy; override;
    procedure Clear; override;
    procedure Add(NewDim: TEssDimension);
    property Dimension[Index: Integer]: TEssDimension read get write put; default;
  end;

{******************************************************************************}
{TEssDatabase -  database
{******************************************************************************}
  TEssDatabase = class
  private
    FName: string;
    FOwner: TEssDBConnection;
    FDimensions: TEssDimensionList;
  protected
  public
    constructor Create(AOwner: TEssDBConnection); virtual;
    destructor Destroy; override;
    function Member(MName: string): TEssMemberBase;
    function DimCount: integer;
    property Dimensions: TEssDimensionList read FDimensions;
  end;

var
  EsbSession: TEssSession;

implementation

uses Dlistdb;
{******************************************************************************}
{TEssSession - an instance of the api}
{******************************************************************************}
constructor TEssSession.Create(AOwner: TComponent);
const
  Registered: Boolean = False;
begin

  inherited Create(AOwner);

{  FSvrConnections := TList.Create;}
  FMaxHandles := 0;
  FMaxBuffer := 0;
  FLocalPath[0] := #0;
  FMessagePath[0] := #0;
  Start;

  {need to register report classes to allow
   streaming in IDE (saving object info in dfm)}
{
  if not Registered then
    begin
      RegisterClasses([TEssReport,
                       TEssReportMbr]);
      Registered := True;
    end;
 }
end;

destructor TEssSession.Destroy;
begin

  Stop;

  while FSvrConnections.Count > 0 do
    RemoveConnection(FSvrConnections.Last);

  FSvrConnections.Free;
  FSvrConnections := nil;

  inherited Destroy;

end;

procedure TEssSession.RemoveConnection(Connection: TEssSvrConnection);
begin
  FSvrConnections.Remove(Connection);
end;

function TEssSession.getLocalPath: string;
begin
  Result := StrPas(FLocalPath);
end;

procedure TEssSession.SetLocalPath(PathName: string);
begin
   StrPLCopy(FLocalPath,PathName,length(PathName));
end;

function TEssSession.getMessagePath: TFileName;
begin
  Result := StrPas(FMessagePath);
end;

procedure TEssSession.SetMessagePath(PathName: TFileName);
begin
   StrPLCopy(FMessagePath,PathName,length(PathName));
end;

function TEssSession.getStatus: Boolean;
begin
  result := FInstance <> 0;
end;

procedure TEssSession.setStatus(Option: Boolean);
begin
  if Option then
    Start
  else Stop;
end;

procedure TEssSession.Start;
var
  InitStruct: ESS_INIT_T;
  retval: ESS_STS_T;
begin

  if Finstance <> 0 then
    Exit;
  {raise an error if already started?}
  if Assigned(FBeforeStart) then
    FBeforeStart(Self);
  with InitStruct do begin
    Version := ESS_API_VERSION;
    UserContext :=  nil;  (* void pointer to user's message context *)
    MaxHandles :=  FMaxHandles; (* max number of context handles required *)
    MaxBuffer :=   FMaxBuffer;   (* max size of buffer that can be allocated *)
    if StrLen(FLocalPath) > 0 then
      LocalPath := FLocalPath    (* local path to use for file operations *)
    else LocalPath := nil;
    if StrLen(FMessagePath) > 0 then
      MessageFile := FMessagePath
    else MessageFile := nil;    (* full path name of message database file *)
    AllocFunc := nil;  (* user-defined memory allocation function *)
    ReallocFunc := nil;  (* user-defined memory reallocation function *)
    FreeFunc := nil;  (* user-defined memory free function *)
    MessageFunc := nil;  (* user-defined message callback function *)
    HelpFile := nil;    (* user-defined help file path *)
    {Ess_System := nil; } (* reserved for internal use *)
  end;

  RetVal := EssInit(@InitStruct, @FInstance);
  if retval <> ESS_STS_NOERR then
    raise EessInitErr.Create('Could not initialize Essbase Library.'
      + #13 + #10 + '(' + IntToStr(retval) + ')');

  if Assigned(FAfterStart) then
    FAfterStart(Self);
end;

procedure TEssSession.Stop;
var
  i: integer;
  tmpInstance: ESS_HINST_T;
begin

  if FInstance = 0 then
    exit;
{
    for i := 0 to FSvrConnections.Count - 1 do begin
      if FSvrConnections[i] <> nil then
        if (TObject(FSvrConnections[i]) is TEssSvrConnection) then
          TEssSvrConnection(FSvrConnections[i]).Connected := false;
    end;
 }
  tmpInstance := FInstance;
  FInstance := 0;
  EssTerm(tmpInstance);

end;

{******************************************************************************}
{TEssSvrConnection - an instance of the api}
{******************************************************************************}
{    FServer: ESS_SVRNAME_T;
    FUserName: ESS_USERNAME_T;
    FPassword: ESS_PASSWORD_T;
    FAppList: TStringList;
    FDBList: TStringList;
    FContext: ESS_HCTX_T;
    FSaveLoginInfo: Boolean;
    FSavePassword: Boolean;
    FDBConnections.TList;
}
constructor TEssSvrConnection.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  EsbSession.FSvrConnections.Add(Self);
  FDBConnections := TList.Create;
  FAppList := TStringList.Create;
  FDBList := TStringList.Create;
  FAppDBList := TStringList.Create;
  FContext := 0;
  FUserName[0] := #0;
  FPassword[0] := #0;
end;

destructor TEssSvrConnection.Destroy;
begin
  Disconnect;
  while FDBConnections.Count > 0 do
    RemoveDBConnection(FDBConnections.Last);
  FDBConnections.Free;
  FAppList.Free;
  FDBList.Free;
  FAppDbList.Free;
  EsbSession.FSvrConnections.Remove(Self);
  inherited Destroy;
end;

procedure TEssSvrConnection.RemoveDBConnection(DBConnection: TEssDBConnection);
begin
  DBConnection.SvrConnection := nil;
  FDBConnections.Remove(DBConnection);
end;

function TEssSvrConnection.getConnected: Boolean;
begin
  result := FContext <> 0;
end;

function TEssSvrConnection.getPassword: string;
begin
  result := StrPas(FPassword);
end;

function TEssSvrConnection.getSavePassword: Boolean;
begin
  result  := (FSavePassword and FSaveLoginInfo);
end;

function TEssSvrConnection.getServer: string;
begin
  result := StrPas(FServer);
end;

function TEssSvrConnection.getUserName: string;
begin
  result := StrPas(FUserName);
end;

function TEssSvrConnection.getInstance: ESS_HINST_T;
begin
  if EsbSession <> nil then
    Result := EsbSession.Instance
  else Result := 0;
end;

procedure TEssSvrConnection.setConnected(Value: Boolean);
begin
  if Value then
    Connect
  else Disconnect;
end;

procedure TEssSvrConnection.setPassword(Value: string);
begin
  if CompareText(Value, Password) = 0 then
    exit;
  Disconnect;
  StrPLCopy(FPassword,Value,length(Value));
end;

procedure TEssSvrConnection.setSaveLoginInfo(Value: Boolean);
begin
  FSaveLoginInfo := Value;
  if not FSaveLoginInfo then
    FSavePassword := False;
end;

procedure TEssSvrConnection.setSavePassword(Value: Boolean);
begin
  if FSaveLoginInfo then
    FSavePassword := Value;
end;

procedure TEssSvrConnection.setServer(Value: string);
begin
  if CompareText(Value, Server) = 0 then
    exit;
  Disconnect;
  StrPLCopy(FServer,Value,length(Value));
end;

procedure TEssSvrConnection.setUserName(Value: string);
begin
  if CompareText(Value, UserName) = 0 then
    exit;
  Disconnect;
  StrPLCopy(FUserName,Value,length(Value));
end;

procedure TEssSvrConnection.connect;
var
{  d: TdlgLogon; }
  retval: LongInt;
{  i: integer;
  s: ESS_USHORT_T;
  DbCount: ESS_USHORT_T;
  pDbList: ESS_PAPPDB_T;      }
  Access: ESS_ACCESS_T;
begin

  if FContext <> 0 then
    {we're already connected}
    exit;

  if EsbSession = nil then
    raise EEssErr.Create('No session Exists');

  if (not EsbSession.Started) then
    raise EEssErr.Create('No API session started.');

  if not FSaveLoginInfo then begin
    FServer[0] := #0;
    FUserName[0] := #0;
  end;

  if not FSavePassword then
    FPassword[0] := #0;

{      retval := EssLogin(EsbSession.Instance,
                     FServer,
                     FUserName,
                     FPassword,
                     @DbCount,
                     @pDbList,
                     @FContext);    }
  retval := EssAutoLogin(EsbSession.Instance,
                     FServer,
                     FUserName,
                     FPassword,
                     nil,
                     nil,
                     AUTO_NOSELECT,
                     @Access,
                     @FContext);
  if retval <> ESS_STS_NOERR then begin
    if retval <> ESS_STS_CANCEL then
      raise EEssErr.Create('Log on failed.'
        + #13 + #10 + '(' + IntToStr(retval) + ')');
      abort;
  end;
    {
  d := TdlgLogon.Create(Application);
  try
    d.Server := Server;
    d.UserName := UserName;
    d.Password := Password;
    d.ShowModal;
    Screen.Cursor := crHourglass;
    try

      if not d.modified then
        exit;

      Server := d.Server;
      UserName := d.UserName;
      Password := d.Password;

      FAppList.Clear;
      FDBList.Clear;
      FAppDbList.Clear;
      pDbList := nil;

      retval := EssLogin(EsbSession.Instance,
                     FServer,
                     FUserName,
                     FPassword,
                     @DbCount,
                     @pDbList,
                     @FContext);
      if retval <> ESS_STS_NOERR then begin
        if retval <> ESS_STS_CANCEL then
          raise EEssErr.Create('Log on failed.'
            + #13 + #10 + '(' + IntToStr(retval) + ')');
        abort;
      end;

      for s := 0 to DbCount - 1 do begin
        if pDbList <> nil then
          if (pDbList^.AppName <> #0) and (pDbList^.DbName <> #0) then begin
            FAppList.Add(StrPas(pDbList^.AppName));
            FDBList.Add(StrPas(pDbList^.DBName));
            FAppDBList.Add(FAppList[FAppList.Count -1] + ': ' + FDbList[FDbList.Count -1]);
          end;
        inc(pDbList);
      end;

      for i := 0 to FDBConnections.Count -1 do begin
        if FDBConnections[i] <> nil then
          if TObject(FDBConnections[i]) is TEssDBConnection then
            TEssDBConnection(FDBConnections[i]).doConnect(True);
      end;

    finally
      Screen.Cursor := crDefault;
    end;

  finally
    d.Free;
  end;
}

{
  retval := EssAutoLogin(EsbSession.Instance,
                         FServer,
                         FUserName,
                         FPassWord,
                         FAppName,
                         FDBName,
                         ESS_USHORT_T(FOptions),
                         @FAccess,
                         @FContext);
 }

end;

procedure TEssSvrConnection.disconnect;
var
  i: integer;
begin
  Screen.Cursor := crHourglass;

  try

    for i := 0 to FDBConnections.Count -1 do begin
      if FDBConnections[i] <> nil then
        if TObject(FDBConnections[i]) is TEssDBConnection then
          TEssDBConnection(FDBConnections[i]).doConnect(False);
    end;

    if FContext = 0 then
      exit;
    EssLogOut(FContext);
    FContext := 0;

  finally
    Screen.Cursor := crDefault;
  end;

end;

{******************************************************************************}
{TEssMemberBase - Base Class for members and dimensions}
{******************************************************************************}
constructor TEssMemberBase.Create(AOwner: TEssDBConnection; ObjName: string);
begin
  FName := ObjName;
  FAlias := '';
  FOwner := AOwner;
  FMemberList := TEssMemberList.Create;
end;

destructor TEssMemberBase.Destroy;
begin
  FMemberList.Free;
  inherited Destroy;
end;

function TEssMemberBase.getAlias: string;
begin
  if length(FAlias) = 0 then
    Result := FName
  else Result := FAlias;
end;

function TEssMemberBase.mbrCount: integer;
begin
  Result := FMemberList.Count;
end;

function TEssMemberBase.HasChildren: Boolean;
begin
  Result := FMemberList.Count > 0;
end;

function TEssMemberBase.Member(MName: string): TEssMemberBase;
var
  i: integer;
begin
  Result := nil;
  if MName = FName then
    Result := Self
  else
  for i := 0 to Members.Count - 1 do begin
    Result := Members[i].member(MName);
    if Result <> nil then
      break;
  end;
end;

function TEssMemberBase.isDescendant(MName: string): Boolean;
var
  tmpStrList: TStringList;
  i: integer;
begin
  tmpStrList := TStringList.Create;
  try
    {get the descendants}
    Descendants(tmpStrList);
    Result := tmpStrList.IndexOf(MName) >= 0;
  finally
    tmpStrList.Free;
  end;
end;

procedure TEssMemberBase.AllInSameDim(FillList: TStringList);
var
  i: integer;
  tmpStrList: TStringList;
begin
 {
  if FOwner.QueryForMembers then begin
 }
    tmpStrList := TStringList.Create;
    try
      tmpStrList.Add('<NEWLINESEPARATED');
      tmpStrList.Add(' <MBRNAMES');
      tmpStrList.Add(' <ALLINSAMEDIM "' + FName + '"');
      FOwner.QueryMemberData(tmpStrList.GetText, FillList);
    finally
      tmpStrList.Free;
    end;
{  end
  else begin
  end;
}
end;

procedure TEssMemberBase.AllSiblings(FillList: TStringList);
var
  i: integer;
  tmpStrList: TStringList;
begin
  {
  if FOwner.QueryForMembers then begin
}
    tmpStrList := TStringList.Create;
    try
      tmpStrList.Add('<NEWLINESEPARATED');
      tmpStrList.Add(' <MBRNAMES');
      tmpStrList.Add(' <ALLSIBLINGSOF "' + FName + '"');
      FOwner.QueryMemberData(tmpStrList.GetText, FillList);
    finally
      tmpStrList.Free;
    end;
{  end
  else begin
  end;
}
end;

procedure TEssMemberBase.Ancestors(FillList: TStringList);
var
  i: integer;
  tmpStrList: TStringList;
begin
{
  if FOwner.QueryForMembers then begin
}
    tmpStrList := TStringList.Create;
    try
      tmpStrList.Add('<NEWLINESEPARATED');
      tmpStrList.Add(' <MBRNAMES');
      tmpStrList.Add(' <ANCESTORSOF "' + FName + '"');
      FOwner.QueryMemberData(tmpStrList.GetText, FillList);
    finally
      tmpStrList.Free;
    end;
{  end
  else begin
  end;
}
end;

procedure TEssMemberBase.Children(FillList: TStringList);
var
  i: integer;
  tmpStrList: TStringList;
begin

  if FOwner.QueryForMembers then begin
    tmpStrList := TStringList.Create;
    try
      tmpStrList.Add('<NEWLINESEPARATED');
      tmpStrList.Add(' <MBRNAMES');
      tmpStrList.Add(' <CHILDRENOF "' + FName + '"');
      FOwner.QueryMemberData(tmpStrList.GetText, FillList);
    finally
      tmpStrList.Free;
    end;
  end
  else begin
    for i := 0 to Members.Count - 1 do
      FillList.AddObject(members[i].Name,members[i]);
  end;
end;

procedure TEssMemberBase.Descendants(FillList: TStringList);
var
  i: integer;
  tmpStrList: TStringList;
begin

  if FOwner.QueryForMembers then begin
    tmpStrList := TStringList.Create;
    try
      tmpStrList.Add('<NEWLINESEPARATED');
      tmpStrList.Add(' <MBRNAMES');
      tmpStrList.Add(' <DescendantsOF "' + FName + '"');
      FOwner.QueryMemberData(tmpStrList.GetText, FillList);
    finally
      tmpStrList.Free;
    end;
  end
  else begin
    for i := 0 to Members.Count - 1 do begin
      FillList.AddObject(Members[i].Name,Members[i]);
      members[i].Descendants(FillList);
    end;
  end;
end;

procedure TEssMemberBase.DimBottom(FillList: TStringList);
var
  i: integer;
  tmpStrList: TStringList;
begin
{
  if FOwner.QueryForMembers then begin
}
    tmpStrList := TStringList.Create;
    try
      tmpStrList.Add('<NEWLINESEPARATED');
      tmpStrList.Add(' <MBRNAMES');
      tmpStrList.Add(' <DIMBOTTOM "' + FName + '"');
      FOwner.QueryMemberData(tmpStrList.GetText, FillList);
    finally
      tmpStrList.Free;
    end;
{  end
  else begin
  end;
}
end;

procedure TEssMemberBase.DimTop(FillList: TStringList);
var
  i: integer;
  tmpStrList: TStringList;
begin
{
  if FOwner.QueryForMembers then begin
}
    tmpStrList := TStringList.Create;
    try
      tmpStrList.Add('<NEWLINESEPARATED');
      tmpStrList.Add(' <MBRNAMES');
      tmpStrList.Add(' <DIMTOP "' + FName + '"');
      FOwner.QueryMemberData(tmpStrList.GetText, FillList);
    finally
      tmpStrList.Free;
    end;
{  end
  else begin
  end;
}
end;

procedure TEssMemberBase.IAncestors(FillList: TStringList);
var
  i: integer;
  tmpStrList: TStringList;
begin
{
  if FOwner.QueryForMembers then begin
}
    tmpStrList := TStringList.Create;
    try
      tmpStrList.Add('<NEWLINESEPARATED');
      tmpStrList.Add(' <MBRNAMES');
      tmpStrList.Add(' <CHILDRENOF "' + FName + '"');
      FOwner.QueryMemberData(tmpStrList.GetText, FillList);
    finally
      tmpStrList.Free;
    end;
    FillList.Add(FName);
{  end
  else begin
  end;
}
end;

procedure TEssMemberBase.IChildren(FillList: TStringList);
var
  i: integer;
  tmpStrList: TStringList;
begin

  if FOwner.QueryForMembers then begin
    tmpStrList := TStringList.Create;
    try
      tmpStrList.Add('<NEWLINESEPARATED');
      tmpStrList.Add(' <MBRNAMES');
      tmpStrList.Add(' <CHILDRENOF "' + FName + '"');
      FOwner.QueryMemberData(tmpStrList.GetText, FillList);
    finally
      tmpStrList.Free;
    end;
    FillList.Add(FName);
  end
  else begin
    Children(FillList);
    FillList.AddObject(Fname,Self);
  end;
end;

procedure TEssMemberBase.IDescendants(FillList: TStringList);
var
  i: integer;
  tmpStrList: TStringList;
begin

  if FOwner.QueryForMembers then begin

    tmpStrList := TStringList.Create;
    try
      tmpStrList.Add('<NEWLINESEPARATED');
      tmpStrList.Add(' <MBRNAMES');
      tmpStrList.Add(' <CHILDRENOF "' + FName + '"');
      FOwner.QueryMemberData(tmpStrList.GetText, FillList);
    finally
      tmpStrList.Free;
    end;
    FillList.Add(FName);
  end
  else begin
    Descendants(FillList);
    FillList.AddObject(Fname,Self);
  end;

end;

procedure TEssMemberBase.OfSameGen(FillList: TStringList);
var
  i: integer;
  tmpStrList: TStringList;
begin
{
  if FOwner.QueryForMembers then begin
}
    tmpStrList := TStringList.Create;
    try
      tmpStrList.Add('<NEWLINESEPARATED');
      tmpStrList.Add(' <MBRNAMES');
      tmpStrList.Add(' <CHILDRENOF "' + FName + '"');
      FOwner.QueryMemberData(tmpStrList.GetText, FillList);
    finally
      tmpStrList.Free;
    end;
{  end
  else begin
  end;
}
end;

procedure TEssMemberBase.OnSameLevel(FillList: TStringList);
var
  i: integer;
  tmpStrList: TStringList;
begin
{
  if FOwner.QueryForMembers then begin
}
    tmpStrList := TStringList.Create;
    try
      tmpStrList.Add('<NEWLINESEPARATED');
      tmpStrList.Add(' <MBRNAMES');
      tmpStrList.Add(' <CHILDRENOF "' + FName + '"');
      FOwner.QueryMemberData(tmpStrList.GetText, FillList);
    finally
      tmpStrList.Free;
    end;
{  end
  else begin
  end;
}
end;

procedure TEssMemberBase.Parent(FillList: TStringList);
var
  i: integer;
  tmpStrList: TStringList;
begin
{
  if FOwner.QueryForMembers then begin
}
    tmpStrList := TStringList.Create;
    try
      tmpStrList.Add('<NEWLINESEPARATED');
      tmpStrList.Add(' <MBRNAMES');
      tmpStrList.Add(' <CHILDRENOF "' + FName + '"');
      FOwner.QueryMemberData(tmpStrList.GetText, FillList);
    finally
      tmpStrList.Free;
    end;
{  end
  else begin
  end;
}
end;

{******************************************************************************}
{TEssMember - an database dimension}
{******************************************************************************}
constructor TEssMember.Create(AOwner: TEssDBConnection; memName, memDimName: string;
      memLevel, memGeneration: ESS_SHORT_T);
begin
  inherited Create(AOwner, memName);
  FInitialized := False;
  FParentMbrName := '';
  FChildMbrName := '';
  FPrevMbrName := '';
  FNextMbrName := '';
  FDescription := '';
  FDimName := memDimName;
  FLevel := memLevel;
  FGeneration := memGeneration;
end;

destructor TEssMember.Destroy;
begin
  inherited Destroy;
end;

procedure TEssMember.Initialize;
var
  retval: ESS_STS_T;
  memName: ESS_MBRNAME_T;
  mbrInfo: ESS_PMEMBERINFO_T;
begin
  if FInitialized then
    exit;
  StrPLCopy(memName,FName,length(FName));
  retval := EssGetMemberInfo(FOwner.ContextHandle, memName,@mbrInfo);
  try
    FParentMbrName := strPas(mbrInfo^.ParentMbrName);
    FChildMbrName := strPas(mbrInfo^.ChildMbrName);
    FPrevMbrName := strPas(mbrInfo^.PrevMbrName);
    FNextMbrName := strPas(mbrInfo^.NextMbrName);
    FDescription := strPas(mbrInfo^.Description);
    FInitialized := True;
  finally
    EssFree(FOwner.Instance,@mbrInfo);
  end;
end;

function TEssMember.HasChildren: Boolean;
begin
  result := FLevel > 0;
end;
(*
function TEssMember.InitChildren(Refresh: Boolean): Boolean;
var
  tmpMember: TEssMember;
  retval: ESS_STS_T;
  mbrInfo: ESS_PMEMBERINFO_T;
  i: integer;
  tmpStr: String;
  ptmpStr: pChar;
  pretStr: pChar;
  memName: ESS_MBRNAME_T;
  tmpStrList: TStringList;
begin

  result := False;

  if FLevel = 0 then
    {there are no children}
    Exit;

  if (FMemberList.Count > 0) and (not refresh) then
    {we already have children, and caller doesn't want to refresh}
    exit;

  if not CheckConnection(FOwner) then
    {make sure we have a database connection}
    exit;

  FMemberList.Clear;
  tmpStrList := TStringList.Create;
  try
    tmpStr := '<CHILDRENOF "' + FName + '"';
    ptmpStr := StrAlloc(Length(tmpStr)+1);
    try
      StrPCopy(ptmpStr,tmpStr);
      retval := EssQueryDataBaseMembers(FOwner.ContextHandle,ptmpStr);
      if retval = ESS_STS_NOERR then begin
        pretStr := nil;
        {dump results into a stringlist. Each name is separated by a tab}
        retval := EssGetString(FOwner.ContextHandle, @pretStr);
        while (retval = ESS_STS_NOERR) and (pretStr <> nil) do begin
          ParsePLine(pretStr,tmpStrList,#9);
          essfree(FOwner.Instance,pointer(pretstr));
          pretStr := nil;
          retval := EssGetString(FOwner.ContextHandle, @pretStr);
        end;
        for i := 0 to tmpStrList.Count -1 do begin
          {get first level of members for dimemsions}
          StrPLCopy(memName,tmpStrList[i],length(tmpStrList[i]));
          retval := EssGetMemberInfo(FOwner.ContextHandle, memName,@mbrInfo);
          if retval = ESS_STS_NOERR then begin
            tmpMember := TEssMember.Create(FOwner, mbrInfo^.mbrName, mbrInfo^.DimName,
                                           mbrInfo^.Level, mbrInfo^.Generation);
            FMemberList.Add(tmpMember);
          end;
        end;
      end;
    finally
      StrDispose(ptmpStr);
    end;
  finally
    tmpStrList.Free;
  end;
  result := true;
end;
*)
{******************************************************************************}
{TEssMemberList -  List of members
{******************************************************************************}
destructor TEssMemberList.Destroy;
begin
  Clear;
  inherited Destroy;
end;

procedure TEssMemberList.Clear;
var
  i: integer;
begin
  for i := 0 to Count -1 do begin
    if Self[i] <> nil then begin
      Self[i].Free;
      Self[i] := nil;
    end;
  end;
  inherited Clear;
end;

function TEssMemberList.Get(Index: Integer): TEssMember;
begin
  Result := TEssMember(inherited get(Index));
end;

procedure TEssMemberList.Put(Index: Integer; Item: TEssMember);
begin
  inherited Put(Index,Item);
end;

procedure TEssMemberList.Add(NewMember: TEssMember);
begin
  inherited Add(NewMember);
end;

{******************************************************************************}
{TEssDimension - an database dimension}
{******************************************************************************}
constructor TEssDimension.Create(AOwner: TEssDBConnection; DimStats: ESS_DIMSTATS_T);
var
  tmpMember: TEssMember;
  i, x: integer;
  tmpStrList: TStringList;
  NameList: TStringList;
  LevelList: TStringList;
  GenList: TStringList;
  AliasList: TStringList;
  FMaxGen: integer;
  tmpMemberList: TEssMemberList;

  procedure AddChildren(Parent: TEssMemberBase; GenX: integer);
  var
    i, EndIdx, StartIdx: integer;

  begin

{    find the index for the parent in name list}
    EndIdx := NameList.IndexOf(Parent.Name) - 1;
    {Find the first possible ChildIndex}
    for i := EndIdx downto 0 do begin
      if (StrToInt(GenList[i]) = GenX) then begin
        StartIdx := i + 1;
        break;
      end;
    end;

    for i := StartIdx to EndIdx do begin
      if StrToInt(GenList[i]) = (GenX + 1) then begin
        tmpMember := TEssMember.Create(FOwner, NameList[i], FName,
                                     StrToInt(LevelList[i]), StrToInt(GenList[i]));

        tmpMember.FAlias := AliasList[i];
        tmpMember.FParentMbrName := Parent.Name;
        Parent.FMemberList.Add(tmpMember);
        if Parent is TEssMember then
          if Length(TEssMember(Parent).FChildMbrName) = 0 then
            TEssMember(Parent).FChildMbrName := NameList[i];
      end;
    end;
    for i := 0 to Parent.FMemberList.Count - 1 do begin
      if (i > 0) then
        Parent.FMemberList[i].FPrevMbrName := NameList[i-1];
      if i < NameList.Count - 1 then
        Parent.FMemberList[i].FNextMbrName := NameList[i+1];
    end;
  end;

begin
  inherited Create(AOwner, StrPas(DimStats.DimName));
  FDimName := FName;
  FDeclDimSize := DimStats.DeclaredDimSize;
  FActDimSize := DimStats.ActualDimSize;
  NameList := TStringList.Create;
  LevelList := TStringList.Create;
  GenList := TStringList.Create;
  AliasList := TStringList.Create;
  tmpStrList := TStringList.Create;
  tmpMemberList := TEssMemberList.Create;

  try

    {Query member data into parallel lists}
    tmpStrList.Add('<NEWLINESEPARATED');
    tmpStrList.Add(' <MBRNAMES');
    tmpStrList.Add(' <DESCENDANTSOF "' + FName + '"');
    FOwner.QueryMemberData(tmpStrList.GetText, NameList);

    tmpStrList[1] := ' <LEVELNUMBERS';
    FOwner.QueryMemberData(tmpStrList.GetText, LevelList);

    tmpStrList[1] := ' <GENERATIONS';
    FOwner.QueryMemberData(tmpStrList.GetText, GenList);

    tmpStrList[1] := ' <ALTNAMES';
    FOwner.QueryMemberData(tmpStrList.GetText, AliasList);

    {Fill Dimension member list}
    FMaxGen := 2;
    for i := 0 to NameList.Count - 1 do begin

      if strtoint(GenList[i]) > FMaxGen then
        FMaxGen := strtoint(GenList[i]);

      if StrToInt(GenList[i]) = 2 then begin
        tmpMember := TEssMember.Create(FOwner, NameList[i], FName,
                                     StrToInt(LevelList[i]), StrToInt(GenList[i]));
        tmpMember.FAlias := AliasList[i];
        tmpMember.FParentMbrName := FName;
        if (i > 0) then
          tmpMember.FPrevMbrName := NameList[i-1];
        if i < NameList.Count - 2 then
          tmpMember.FNextMbrName := NameList[i+1];
        FMemberList.Add(tmpMember);
      end
      else FLevel := StrToInt(LevelList[i]);
    end;

    {Fill member lists for all members}
    for i := 2 to FMaxGen - 1 do
      for x := 0 to NameList.Count - 1 do begin
        if (StrToInt(GenList[x]) = i) and (StrToInt(LevelList[x]) > 0) then
          AddChildren(member(NameList[x]), i);
    end;

  finally
    essFree(FOwner.Instance,@DimStats);
    tmpStrList.Free;
    NameList.Free;
    LevelList.Free;
    GenList.Free;
    AliasList.Free;
    tmpMemberList.Free;
  end;
end;

destructor TEssDimension.Destroy;
begin
  inherited Destroy;
end;

{******************************************************************************}
{TEssDDimensionList -  database
{******************************************************************************}
destructor TEssDimensionList.Destroy;
begin
  Clear;
  inherited Destroy;
end;

procedure TEssDimensionList.Clear;
var
  i: integer;
begin
  for i := 0 to Count -1 do begin
    if Self[i] <> nil then begin
      Self[i].Free;
      Self[i] := nil;
    end;
  end;
  inherited Clear;
end;

function TEssDimensionList.Get(Index: Integer): TEssDimension;
begin
  Result := TEssDimension(inherited get(Index));
end;

procedure TEssDimensionList.Put(Index: Integer; Item: TEssDimension);
begin
  inherited Put(Index,Item);
end;

procedure TEssDimensionList.Add(NewDim: TEssDimension);
begin
  inherited Add(NewDim);
end;

{******************************************************************************}
{TEssDatabase -  database
{******************************************************************************}
constructor TEssDatabase.Create(AOwner: TEssDBConnection);
var
  tmpDimension: TEssDimension;
  pDbStats: ESS_PDBSTATS_T;
  retval: ESS_STS_T;
  nDims, i: Longint;
begin
  inherited Create;
  FName := '';
  FOwner := AOwner;
  FDimensions := TEssDimensionList.Create;
  with FOwner do begin

    {make sure we're connected to a specific database, not just the server}
    if Length(AppName) = 0 then
      abort;
    if (Length(DbName) = 0) then
      abort;

    FName := DbName;
    retval := EssGetDataBaseStats(ContextHandle,FAppName,FDbName,@pDbStats);
    if retval <> ESS_STS_NOERR then
      abort;
    try
      nDims := pDbStats^.nDims;
      for i := 0 to nDims - 1 do begin
        tmpDimension := TEssDimension.Create(FOwner, pDbStats^.DimStatsAry[i]);
        FDimensions.Add(tmpDimension);
        if i > 0 then
          tmpDimension.FPrevMbrName := FDimensions[i - 1].Name;
        if i < (nDims -1) then
          tmpDimension.FNextMbrName := StrPas(pDbStats^.DimStatsAry[i + 1].DimName);
      end;
    finally
      EssFree(Instance,pointer(pDbStats));
    end;
  end;
end;

destructor TEssDatabase.Destroy;
begin
  FDimensions.Free;
  inherited Destroy;
end;

function TEssDatabase.DimCount: integer;
begin
  Result := FDimensions.Count;
end;

function TEssDatabase.Member(MName: string): TEssMemberBase;
var
  i: integer;
begin
  Result := nil;
  for i := 0 to FDimensions.Count - 1 do begin
    Result := FDimensions[i].member(MName);
    if Result <> nil then
      break;
  end;
end;

{******************************************************************************}
{TEsbDataSource - EsbDataSource object. Used to communicate with data controls}
{******************************************************************************}
constructor TEssDataSource.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FOnConnect := nil;
  FOnDisconnect := nil;
  FConnection := nil;
end;

procedure TEssDataSource.GetDimensionList(Sender: TComponent; List: TStrings);
begin
  if FConnection <> nil then
    FConnection.GetDimensionList(Sender, List);
end;

function TEssDataSource.Member(MName: string): TEssMemberBase;
begin
  if FConnection = nil then
    Result := nil
  else Result := FConnection.Member(MName);
end;

destructor TEssDataSource.destroy;
begin
  if (FConnection <> nil) then
    FConnection.FEssDataSources.Remove(Self);
  inherited Destroy;
end;

procedure TEssDataSource.doConnect(Value: Boolean);
begin
  if Value then begin
    if assigned(FOnConnect) then
      FOnConnect(nil);
  end
  else begin
    if assigned(FOnDisconnect) then
      FOnDisconnect(nil);
  end;
end;

function TEssDataSource.getConnection: TEssDBConnection;
begin
  Result := FConnection;
end;

procedure TEssDataSource.setConnection(NewConnection: TEssDBConnection);
begin

  if (NewConnection = FConnection) then
    exit;
  doConnect(False);
  if (FConnection <> nil) then
    FConnection.FEssDataSources.Remove(Self);
  if NewConnection <> nil then
    NewConnection.FEssDataSources.Add(Self);
  FConnection := NewConnection;

end;

{******************************************************************************}
{TEssDBConnection - an instance of the api}
{******************************************************************************}
constructor TEssDBConnection.Create(AOwner: TComponent);
var
  I: Integer;
  Component: TComponent;
begin
  inherited Create(AOwner);

  FDatabase := nil;
  FSvrConnection := nil;
  FShowDbList := False;
  FConnected := False;
  FAppDbIndex := -1;
  FOptions := alDefault;
  FAccess := ESS_PRIV_NONE;
  FConnected := False;
  FAppName[0] := #0;
  FDbName[0] := #0;
  FQueryForMembers := False;
  FEssDataSources := TList.Create;

end;

procedure TEssDBConnection.GetDimensionList(Sender: TComponent; List: TStrings);
var
  i: integer;
begin
  if FDatabase <> nil then begin
    for i := 0 to FDatabase.Dimensions.Count - 1 do
      List.Add(FDatabase.Dimensions[i].Name);
  end;

end;

destructor TEssDBConnection.Destroy;
begin

  Disconnect;

  while FEssDataSources.Count > 0 do
    RemoveEssDataSource(FEssDataSources.Last);

  if FDatabase <> nil then begin
    FDataBase.Free;
    FDatabase := nil;
  end;

  if (FSvrConnection <> nil) then
    FSvrConnection.FDBConnections.Remove(Self);

  FSvrConnection := nil;
  FEssDataSources.Free;

  inherited Destroy;

end;

function TEssDBConnection.getSvrConnection: TEssSvrConnection;
begin
  Result := FSvrConnection;
end;

procedure TEssDBConnection.setSvrConnection(NewConnection: TEssSvrConnection);
begin

  if (NewConnection = FSvrConnection) then
    exit;

  Disconnect;

  if (FSvrConnection <> nil) then
    FSvrConnection.FDBConnections.Remove(Self);
  if NewConnection <> nil then
    NewConnection.FDBConnections.Add(Self);
  FSvrConnection := NewConnection;

end;

function TEssDBConnection.getInstance: ESS_HINST_T;
begin
  if EsbSession <> nil then
    Result := EsbSession.Instance
  else Result := 0;
end;

procedure TEssDBConnection.setSaveLoginInfo(Value: Boolean);
begin
  FSaveLoginInfo := Value;
end;

procedure TEssDBConnection.RemoveEssDataSource(EssDataSource: TEssDataSource);
begin
  EssDataSource.Connection := nil;
  FEssDataSources.Remove(EssDataSource);
end;

function TEssDBConnection.getContext: ESS_HCTX_T;
begin
  if FSvrConnection = nil then
    Result := 0
  else Result := FSvrConnection.FContext;
end;

function TEssDBConnection.getServer: string;
begin
  if FSvrConnection = nil then
    Result := ''
  else Result := FSvrConnection.Server;
end;

function TEssDBConnection.getUserName: string;
begin
  if FSvrConnection = nil then
    Result := ''
  else Result := FSvrConnection.UserName;
end;

function TEssDBConnection.getPassword: string;
begin
  if FSvrConnection = nil then
    Result := ''
  else Result := FSvrConnection.PassWord;
end;

function TEssDBConnection.getAppName: string;
begin
  result := StrPas(FAppName);
end;

function TEssDBConnection.getDbName: string;
begin
  result := StrPas(FDbName);
end;

procedure TEssDBConnection.setAppName(Value: string);
begin
  if CompareText(Value, AppName) = 0 then
    exit;
  Disconnect;
  StrPLCopy(FAppName,Value,length(Value));
end;

procedure TEssDBConnection.setDbName(Value: string);
begin
  if CompareText(Value, DbName) = 0 then
    exit;
  Disconnect;
  StrPLCopy(FDbName,Value,length(Value));
end;

procedure TEssDBConnection.setConnected(Value: Boolean);
begin
  if Value then
    Connect
  else Disconnect;
end;

function TEssDBConnection.getConnected: Boolean;
begin
  result := FConnected;
end;

function TEssDBConnection.getInitialized: Boolean;
begin
  result := FDatabase <> nil;
end;

procedure TEssDBConnection.setInitialized(Value: Boolean);
begin

  if Value = (FDatabase <> nil) then
    Exit;

  if Value then begin
    Screen.Cursor := crHourglass;
    try
      FDatabase := TEssDatabase.Create(Self);
    finally
      Screen.Cursor := crDefault;
    end;
  end
  else begin
    FDatabase.Free;
    FDatabase := nil;
  end;

end;

function TEssDBConnection.Member(MName: string): TEssMemberBase;
begin
  if FDatabase = nil then
    Result := nil
  else Result := FDataBase.Member(MName);
end;

procedure TEssDBConnection.doConnect(Value: Boolean);
begin
  {we don't want to connect, just because
  a server connection has been established,
  but we do want to disconnect if it's been severed}
  if not Value and FConnected then
    Disconnect;
(*
  if Value and (FDataBase = nil) then begin
    Screen.Cursor := crHourglass;
    try
      FDatabase := TEssDatabase.Create(Self);
    finally
      Screen.Cursor := crDefault;
    end;
  end
  else begin
    {verify DB objects}
    {if not OK, destroy & create New}
  end;
*)
end;

procedure TEssDBConnection.Connect;
var
  d: TdlgSelectDB;
  retval: LongInt;
  i: integer;
  tmpAppName: ESS_STR_T;
  tmpDbName: ESS_STR_T;
begin

  tmpAppName := nil;
  tmpDbName := nil;

  if FSvrConnection = nil then
    raise EEssErr.Create('No Server Connection Exists');

  {establish server connection if needed}
  if not FSvrConnection.Connected then
    FSvrConnection.Connected := True;

  {Something went wrong, or user cancelled}
  if not FSvrConnection.Connected then
    exit;

  if FConnected then
    if  (FSvrConnection.FAppList.IndexOf(AppName)
        + FSvrConnection.FDbList.IndexOf(DbName)) < 0 then begin
      FConnected := False;
      AppName := '';
      DbName := '';
      FAppDBIndex := -1;
    end;

  if FConnected then begin
    {we've been connected before & have an app & dbname.
    reload the database}

    Screen.Cursor := crHourglass;
    try

      retval := ESSGetActive(FSvrConnection.FContext,
                             @tmpAppName,
                             @tmpDbName,
                             @FAccess);

      if retval <> ESS_STS_NOERR then
        raise EEssErr.Create('could not reconnect with Server.'
          + #13 + #10 + '(' + IntToStr(retval) + ')');

      if (tmpAppName = nil)
         or (tmpDbName = nil)
         or (StrIComp(FAppName,tmpAppName) <> 0)
         or (StrIComp(FDbName,tmpDbName) <> 0) then
      retval := ESSSetActive(FSvrConnection.FContext,
                             FAppName,
                             FDbName,
                             @FAccess);
    finally
      Screen.Cursor := crDefault;
    end;
    if retval <> ESS_STS_NOERR then
      raise EEssErr.Create('could not reconnect with "'
                           + AppName + ': ' + DbName + '".'
                           + #13 + #10
                           + '(' + IntToStr(retval) + ')');
  end;

  {if everything's OK & we've been connected at least once, we don't need
  to rebuild database info.
  We should check it first, though}
  if FConnected then
    Exit;

  if not FSaveLoginInfo then begin
    FAppName[0] := #0;
    FDbName[0] := #0;
    {only want to destroy if we're changing databases}
    if FDatabase <> nil then begin
      FDataBase.Free;
      FDatabase := nil;
    end;
  end;

  if Assigned(FBeforeConnect) then
    FBeforeConnect(Self);

  {get dbname if needed}
  if (StrLen(FAppName) = 0) or (StrLen(FDbName) = 0) or FShowDBList then begin
    AppName := '';
    DbName := '';
    FAppDBIndex := -1;
    d := TdlgSelectDB.Create(Application);
    try
      d.Init(FSvrConnection.FAppDbList);
      d.Selection := FAppDBIndex;
      d.ShowModal;                         
      if d.Modified then begin
        { only want to destroy if we're changing databases}
        if FDatabase <> nil then begin
          FDataBase.Free;
          FDatabase := nil;
        end;
        FAppDBIndex := d.Selection;
        if FAppDBIndex > -1 then begin
          AppName := FSvrConnection.FAppList[FAppDBIndex];
          DBName := FSvrConnection.FDBList[FAppDBIndex];
        end;
      end;
    finally
      d.Free;
    end;
  end;

  if (StrLen(FAppName) = 0) or (StrLen(FDbName) = 0) then
    exit;

  {Create a db
  Screen.Cursor := crHourglass;
  try
    FDatabase := TEssDatabase.Create(Self);
  finally
    Screen.Cursor := crDefault;
  end;
  }
  FConnected := True;

  for i := 0 to FEssDataSources.Count -1 do begin
    if FEssDataSources[i] <> nil then
      if TObject(FEssDataSources[i]) is TEssDataSource then
        TEssDataSource(FEssDataSources[i]).doConnect(FConnected);
  end;
 
  if Assigned(FAfterConnect) then
    FAfterConnect(Self);

end;

procedure TEssDBConnection.Disconnect;
var
  i: integer;
begin
  FConnected := False;

  for i := 0 to FEssDataSources.Count -1 do begin
    if FEssDataSources[i] <> nil then
      if TObject(FEssDataSources[i]) is TEssDataSource then
        TEssDataSource(FEssDataSources[i]).doConnect(FConnected);
  end;

end;

{STICK THIS IN DATABASE OBJECT}
function TEssDBConnection.GetMemberDimension(mbrName: string): string;
var i: integer;
begin
  Result := '';
end;

procedure TEssDBConnection.QueryMemberData(Query: pchar; Dest: TStringList);
var
  retval: ESS_STS_T;
  pretStr: pChar;
  tmpList: TStringList;
begin
  retval := EssQueryDatabaseMembers(ContextHandle, Query);
  if retval <> ESS_STS_NOERR then
    Abort;
  pretStr := nil;
  retval := EssGetString(ContextHandle, @pretStr);
  if retval <> ESS_STS_NOERR then
    abort;
  tmpList := TStringList.Create;
  try
    while (retval = ESS_STS_NOERR) and (pretStr <> nil) do begin
      tmpList.SetText(pretStr);
      try
        Dest.AddStrings(tmpList);
      finally
        essfree(Instance,pointer(pretstr));
        pretStr := nil;
        retval := EssGetString(ContextHandle, @pretStr);
        if retval <> ESS_STS_NOERR then begin
          Dest.Clear;
          abort;
        end;
      end;
    end;
  finally
    tmpList.Free;
  end;
end;

procedure DoneEsbSession; far;
begin
  EsbSession.Free;
end;

begin

  EsbSession := TEssSession.Create(nil);
  AddExitProc(DoneEsbSession);

end.
