{******************************************************************************}
{* Project:                                                                   *}
{* Unit:        dcedit                                                        *}
{* Description: Document and project objects                                  *}
{*                                                                            *}
{* 1996, 1997, Richard L. Chase, All Rights Reserved                         *}
{*                                                                            *}
{* Types:                                                                     *}
{*    TDocument - a basic text document. Handles opening, saving, etc.        *}
{*    TProject - Holds collection of documents & controls their creation      *}
{*               and destuction. Projects can be saved to files               *}
{*                                                                            *}
{* Forms:                                                                     *}
{*    None                                                                    *}
{*                                                                            *}
{* Procedures:                                                                *}
{*    None                                                                    *}
{*                                                                            *}
{* Functions:                                                                 *}
{*    None                                                                    *}
{*                                                                            *}
{******************************************************************************}
unit dcedit;

interface

uses
  SysUtils,
  Windows,
  Classes,
  Registry,
  graphics,
  dialogs;

type

  TProject = class;

  {onNameChange Event}
  TNameChangeEvent = procedure(Sender: TObject; NewName: string) of object;

  TDocument = class(TPersistent)
  private
    FFileName: string;
    FFilePath: string;
    FDocTitle: string;
    FNewDoc: Boolean;
    FDirty: Boolean;
    FProject: TProject;
    FOnNameChange: TNameChangeEvent;
    FisOpen: Boolean;
  protected
    procedure setFileName(NewName: string); virtual;
    procedure setTitle(NewTitle: string);
  public
    constructor Create; virtual;
    constructor CreateOpen(FileName: String); virtual;
    procedure RetrieveContents(Contents: TStrings); virtual;
    procedure Load(FileName: string); virtual;
    procedure Save(SourceText: TStrings); virtual;
    procedure Clear; virtual;
    property Dirty: Boolean read FDirty write FDirty;
    property FileName: string read FFileName write setFileName;
    property FilePath: string read FFilePath write FFilePath;
    property Project: TProject read FProject;
    property Title: string read FDocTitle;
    Property isNew: Boolean read FNewDoc;
    Property isOpen: Boolean read FisOpen write FisOpen;
    property OnNameChange: TNameChangeEvent read FOnNameChange write FOnNameChange;
  end;

  TDocumentList = class(TStringList)
  private
  protected
    function GetDocument(Index: Integer): TDocument; virtual;
    procedure PutDocument(Index: Integer; AProperty: TDocument); virtual;
    procedure PutObject(Index: Integer; AObject: TObject); override;
  public
    constructor Create;
    destructor destroy; override;
{    function Add(const S: string): Integer; override;}
    function AddObject(const S: string; AObject: TObject): Integer; override;
    procedure InsertObject(Index: Integer; const S: string;
      AObject: TObject);
    procedure Delete(Index: Integer); override;
{    procedure Insert(Index: Integer; const S: string); override;}
    property Documents[Index: Integer]: TDocument read GetDocument write PutDocument;
  end;

  TProject = class(TComponent)
  private
    fName: string;
    FFileName: string;
    FNewProject: Boolean;
    FDirty: Boolean;
    FOnDocNameChange: TNotifyEvent;
    function getDocCount: integer;
  protected
    FMemberFiles: TDocumentList;
    procedure setFileName(Value: string);
    procedure SetProjectName(Value: string);
    function getDirty: Boolean;
  public
    constructor Create(AOwner: TComponent); override;
    constructor CreateOpen(AOwner: TComponent; ProjFile: string); virtual;
    destructor destroy; override;
    procedure Clear;
    function InProject(DocName: string): boolean;
    function DocumentFileName(DocName: string): string;
    function GetDocIndex(DocName: string): integer;
    function GetDocName(Index: integer): string;
    procedure Add(Doc: TDocument);
    procedure AddLoad(FileName: string);
    procedure Load;
    procedure Remove(DocName: String);
    function RenameDoc(OldName: string; var NewName: string): TDocument;
    function Retrieve(DocName: String): TDocument;
    procedure Save;
    property DocCount: integer read getDocCount;
    property FileName: string read FFileName write SetFileName;
    property IsDirty: Boolean read getDirty;
    property IsNew: Boolean read FNewProject;
    property Name: string read FName write SetProjectName;
    property OnDocNameChange: TNotifyEvent read FOnDocNameChange write FOnDocNameChange;
  end;


implementation

const

  PROJ_NAMEVERLAST    = 'Dick Chase''' + 's Essbase(TM) Editor 1.0';
  PROJ_NAMEVER    = 'Dick Chase''' + 's Essbase(TM) Editor 1.0';
  PROJ_FILEVER    = '1';
  PROJ_PROJECT    = 'Project';
  PROJ_FILES      = 'Files';

{******************************************************************************}
{**                                                                          **}
{**                              TDocument                                   **}
{**                                                                          **}
{******************************************************************************}
constructor TDocument.Create;
begin
  inherited Create;
  FProject := nil;
  FNewDoc := True;
  FFileName := '';
  FFilePath := '';
  FDirty := False;
  FOnNameChange := nil;
end;

constructor TDocument.CreateOpen(FileName: String);
begin
  if not FileExists(FileName) then
    raise Exception.Create('Can''' + 't find ''' + FileName + '''');
  Create;
  FOnNameChange := nil;
  Load(FileName);
end;

procedure TDocument.Clear;
begin
  FFileName := '';
  FFilePath := '';
end;

procedure TDocument.Load(FileName: string);
var
  i: integer;
begin
  i := -1;
  if not FileExists(FileName) then
    raise Exception.Create('Can''' + 't find ''' + FileName + '''');
  {get the index of doc in project}
  if FProject <> nil then
    i := FProject.GetDocIndex(FFileName);
  Clear;
  FFileName := ExtractFileName(FileName);
  FFilePath := ExtractFilePath(FileName);
  FDocTitle := FFileName;
  FNewDoc := False;
  FDirty := False;
  if Assigned(FOnNameChange) then
    FOnNameChange(Self,FFileName);
  if (i >= 0) then begin
    FProject.FMemberFiles[i] := FFileName;
    if assigned(FProject.FOnDocNameChange) then
      FProject.FOnDocNameChange(FProject);
  end;

end;

procedure TDocument.setTitle(NewTitle: string);
begin
  FDocTitle := NewTitle;
end;

procedure TDocument.setFileName(NewName: string);
var
  tmpFile: TStringList;
begin
  if NewName = FFileName then
    Exit;
  if Length(NewName) = 0 then
    exit;

  {Create new file and delete old}
  tmpFile := TStringList.Create;
  try
    tmpFile.LoadFromFile(FFilePath + FFileName);
    tmpFile.SaveToFile(FFilePath + NewName);
  finally
    tmpFile.Free;
  end;
  DeleteFile(PChar(FFilePath + FFileName));

  {Change the name and notify owner}
  FFileName := NewName;
  if Assigned(FOnNameChange) then
    FOnNameChange(Self,FFileName);

end;

procedure TDocument.Save(SourceText: TStrings);
begin
  SourceText.SaveToFile(FFilePath + FFileName);
  FNewDoc := False;
  FDirty := False;
end;

procedure TDocument.RetrieveContents(Contents: TStrings);
begin
  Contents.LoadFromFile(FFilePath + FFileName);
end;

{******************************************************************************}
{**                                                                          **}
{**                            TDocumentList                                 **}
{**                                                                          **}
{******************************************************************************}
constructor TDocumentList.Create;
begin
  inherited Create;
end;

destructor TDocumentList.destroy;
begin
  Clear;
  inherited Destroy;
end;

function TDocumentList.GetDocument(Index: Integer): TDocument;
begin
  result := (getObject(Index) as TDocument);
end;

procedure TDocumentList.PutDocument(Index: Integer; AProperty: TDocument);
begin
{
  if objects[index] <> nil then begin
    objects[index].free;
    objects[index] := nil;
  end;
}
  putObject(Index, AProperty);
end;

procedure TDocumentList.PutObject(Index: Integer; AObject: TObject);
begin
  if (AObject <> nil) and (not(AObject is TDocument)) then
    raise Exception.Create('not a TDocument');
  inherited PutObject(Index, AObject);
end;

function TDocumentList.AddObject(const S: string; AObject: TObject): Integer;
begin
  if not(AObject is TDocument) then
    raise Exception.Create('not a TDocument');
  result := inherited AddObject(S,AObject);
end;

procedure TDocumentList.InsertObject(Index: Integer; const S: string;
  AObject: TObject);
begin
  if not(AObject is TDocument) then
    raise Exception.Create('not a TDocument');
  inherited Insert(Index, S);
  PutObject(Index, AObject);
end;

procedure TDocumentList.Delete(Index: Integer);
begin

  if Objects[Index] <> nil then begin
    Objects[Index] := nil;
  end;

  inherited Delete(Index);

end;

{******************************************************************************}
{**                                                                          **}
{**                                 TProject                                 **}
{**                                                                          **}
{******************************************************************************}
constructor TProject.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  fName :='New Project';
  FFileName :='';
  FMemberFiles := TDocumentList.Create;
  FNewProject := True;
  FDirty := False;
end;

destructor TProject.destroy;
begin
  Clear;
  FMemberFiles.Free;
  inherited Destroy;
end;

procedure TProject.Clear;
var
  i: integer;
  tmpDoc: TDocument;
begin
  for i := 0 to FMemberFiles.Count - 1 do begin
    if FMemberFiles.documents[i] <> nil then begin
      tmpDoc := FMemberFiles.documents[i];
      tmpDoc.Free;
    end;
  end;
end;

constructor TProject.CreateOpen(AOwner: TComponent; ProjFile: string);
begin

  Create(AOwner);

  if not FileExists(ProjFile) then
    raise Exception.Create('Can''' + 't find ''' + ProjFile + '''');

  FFileName := ProjFile;

  FNewProject := False;

  Load;

  SetCurrentDir(ExtractFileDir(FileName));

end;

procedure TProject.Load;
var
  tmpList: TStringList;
  i, next: integer;
begin

  Clear;

  tmpList := TStringList.Create;
  try

    tmpList.LoadFromFile(FFileName);

    if (tmpList.Count <2) or (tmpList[1] <> PROJ_FILEVER) then
      raise Exception.Create('''' + FFileName + ''' is not a valid project or is from an earlier version.');

    if (tmpList[1] <> PROJ_FILEVER) and (tmpList[0] = PROJ_NAMEVERLAST) then
    {it's an earlier version with compatible file structure}
      tmpList.Insert(1,PROJ_FILEVER);

    Next := 1;
    {get project Name]}
    for i := Next to tmpList.Count - 1 do begin
      inc(Next);
      if Trim(tmpList[i]) = PROJ_PROJECT then begin
        FName := Trim(tmpList[Next]);
        inc(Next);
        break;
      end;
    end;

    {get files}
    for i := Next to tmpList.Count - 1 do begin
      inc(Next);
      if Trim(tmpList[i]) = PROJ_FILES then begin
        break;
      end;
    end;

    for i := Next to tmpList.Count - 1 do begin
      if length(Trim(tmpList[i])) > 0 then
        if fileExists(tmpList[i]) then begin
          AddLoad(tmpList[i]);
        end;
    end;

    FDirty := False;

  finally
    tmpList.Free;
  end;

end;

procedure TProject.Save;
var
  tmpList: TStringList;
  i: integer;
  tmpDocument: TDocument;
begin
  tmpList := TStringList.Create;
  try
    tmpList.Add(PROJ_NAMEVER);
    tmpList.Add(PROJ_FILEVER);
    tmpList.Add(PROJ_PROJECT);
    tmpList.Add('  ' + FName);
    tmpList.Add('');
    tmpList.Add(PROJ_FILES);

    for i := 0 to FMemberFiles.Count - 1 do begin
      if FMemberFiles.documents[i] <> nil then begin
        tmpDocument := FMemberFiles.Documents[i];
        tmpList.Add(tmpDocument.FilePath + tmpDocument.FileName);
      end;
    end;

    tmpList.SaveToFile(FFileName);

    FNewProject := False;
    FDirty := False;

    SetCurrentDir(ExtractFileDir(FileName));

  finally
    tmpList.Free;
  end;
end;

procedure TProject.Add(Doc: TDocument);
begin
  FMemberFiles.AddObject(Doc.FileName,Doc);
  Doc.FProject := Self;
  FDirty := True;
end;

procedure TProject.AddLoad(FileName: string);
var
  tmpDocument: TDocument;
begin

  if not FileExists(FileName) then
    exit;
  {make sure file isn't already included}
  if FMemberFiles.IndexOf(ExtractFileName(FileName)) >=0 then
    exit;

  tmpDocument := TDocument.CreateOpen(FileName);
  Add(tmpDocument);

end;

procedure TProject.Remove(DocName: String);
var
  i: integer;
  tmpDocument: TDocument;
begin
  for i := 0 to FMemberFiles.Count - 1 do begin
    if FMemberFiles[i] = DocName then begin
      tmpDocument := FMemberFiles.Documents[i];
      FMemberFiles.Delete(i);
      if not tmpDocument.isOpen then
        tmpDocument.Free
      else tmpDocument.FProject := nil;
      break;
    end;
  end;

  FDirty := True;

end;

function TProject.GetDocIndex(DocName: string): Integer;
begin
  result := FMemberFiles.indexof(DocName);
end;

function TProject.GetDocName(Index: integer): string;
begin
  result := FMemberFiles[Index];
end;

function TProject.Retrieve(DocName: String): TDocument;
var
  i: integer;
begin
  Result := nil;
  for i := 0 to FMemberFiles.Count - 1 do begin
    if FMemberFiles[i] = DocName then begin
      Result := FMemberFiles.Documents[i];
      Break;
    end;
  end;
end;

procedure TProject.setFileName(Value: string);
begin
  if FFileName = Value then
    exit;
  FFileName := Value;
  FDirty := True;
end;

procedure TProject.setProjectName(Value: string);
begin
  if FName = Value then
    exit;
  FName := Value;
  FDirty := True;
end;

function TProject.InProject(DocName: string): boolean;
begin
  Result := length(DocumentFileName(DocName)) > 0;
end;

function TProject.DocumentFileName(DocName: string): string;
var
  i: integer;
  tmpDocument: TDocument;
begin
  Result := '';
  for i := 0 to FMemberFiles.Count - 1 do begin
    if FMemberFiles[i] = DocName then begin
      if FMemberFiles.documents[i] <> nil then begin
        tmpDocument := FMemberFiles.Documents[i];
        Result := tmpDocument.FilePath + tmpDocument.FileName;
      end;
    end;
  end;
end;

function TProject.getDirty: Boolean;
var
  i: integer;
  tmpDocument: TDocument;
begin
  Result := FDirty;
  if Result then
    exit;
  for i := 0 to FMemberFiles.Count - 1 do begin
    if FMemberFiles.documents[i] <> nil then begin
      tmpDocument := FMemberFiles.Documents[i];
      Result := tmpDocument.Dirty;
      if Result then
        Break;
    end;
  end;
end;

function TProject.RenameDoc(OldName: string; var NewName: string): TDocument;
var
  tmpDocument: TDocument;
begin
  tmpDocument := Retrieve(OldName);
  tmpDocument.FileName := NewName;
  FDirty := True;
  Result := tmpDocument;
end;

function TProject.getDocCount: integer;
begin
  Result := FMemberFiles.Count;
end;

end.
