unit doptions;

interface

uses
  SysUtils, Windows, Messages, Classes, Graphics, Controls,
  StdCtrls, ExtCtrls, Forms, ComCtrls, javaeditdefs, Shellapi,
  Dialogs, dclstview, Clipbrd, Menus, dMultiSave, Tabs, Tabnotbk, dclstbox;

type
  TdlgOptions = class(TForm)
    mnpDocument: TPopupMenu;
    mnpDocumentOpen: TMenuItem;
    mnpDocumentDelete: TMenuItem;
    mnpDocumentCut: TMenuItem;
    mnpDocumentCopy: TMenuItem;
    mnpDocumentPaste: TMenuItem;
    N1: TMenuItem;
    mnpCompile: TMenuItem;
    pnlButtonsBottom: TPanel;
    pnlButtonsRight: TPanel;
    cmdOK: TButton;
    pnlName: TPanel;
    edtProjectName: TEdit;
    Label1: TLabel;
    mnpBrowse: TMenuItem;
    mnpAdd: TMenuItem;
    TabbedNotebook1: TTabbedNotebook;
    Label5: TLabel;
    edtDirectory: TEdit;
    Label3: TLabel;
    edtClassPath: TEdit;
    Bevel1: TBevel;
    chkShowWarn: TCheckBox;
    chkOptimize: TCheckBox;
    chkDebugInfo: TCheckBox;
    chkVerbose: TCheckBox;
    tsDisplay: TTabSet;
    lstDocuments: TDCListBox;
    procedure FormCreate(Sender: TObject);
    procedure cmdOKClick(Sender: TObject);
    procedure FormDestroy(Sender: TObject);
    procedure lstDocumentsFileDrop(Files: TStrings; X, Y: Integer);
    procedure lstDocumentsDblClick(Sender: TObject);
    procedure mnpDocumentPopup(Sender: TObject);
    procedure mnpDocumentOpenClick(Sender: TObject);
    procedure doDelete(Sender: TObject);
    procedure doCopy(Sender: TObject);
    procedure doCut(Sender: TObject);
    procedure doPaste(Sender: TObject);
    procedure lstDocumentsKeyUp(Sender: TObject; var Key: Word;
      Shift: TShiftState);
    procedure mnpAddClick(Sender: TObject);
    procedure FormActivate(Sender: TObject);
    procedure mnpCompileClick(Sender: TObject);
    procedure mnpBrowseClick(Sender: TObject);
    procedure FormClose(Sender: TObject; var Action: TCloseAction);
    procedure edtProjectNameChange(Sender: TObject);
    procedure lstDocumentsDrawItem(Control: TWinControl; Index: Integer;
      Rect: TRect; State: TOwnerDrawState);
    procedure tsDisplayClick(Sender: TObject);
  private
    FLoadingProject: Boolean;
    FBmpList: TList;
    FProject: TProject;
    FJavaList: TStringList;
    FHTMLList: TStringList;
    FTextList: TStringList;
    FOtherList: TStringList;
    procedure setProject(NewProject: TProject);
    procedure InitDocLists;
    procedure DocNameChange(Sender: TObject);
  public
    function isSelection: Boolean;
    procedure DisplayDocs;
    procedure SaveOptions;
    property Project: TProject write setProject;
  end;

{
  TProjectItem = class
  private
    FCaption: string;
    FFileDate: string;
    FFilePath: string;
    FBitmap: TBitmap;
  public
    constructor Create;
    destructor destroy; override;
    property Bitmap: TBitmap read FBitmap write FBitmap;
    property Caption: string read FCaption write FCaption;
    property FileDate: string read FFileDate write FFileDate;
    property FilePath: string read FFilePath write FFilePath;
  end;
}
var
  dlgOptions: TdlgOptions;

implementation

{$R *.DFM}
{$R JEBMPS.RES}

uses fmain;

const
  itBitmap: TResType = Controls.rtBitmap; // Reference for duplicate identifier

{
constructor TProjectItem.Create;
begin
  FBitmap := nil;
end;

destructor TProjectItem.destroy;
begin
  FBitmap.Free;
  FBitmap := nil;
end;
}
procedure TdlgOptions.InitDocLists;
var
  i: integer;
{  NewItem: TProjectItem;

  procedure LoadFileInfo(Item: TProjectItem);
  var
    tmpDocument: TDocument;
    fHandle: THandle;
  begin
    tmpDocument := FProject.Retrieve(Item.Caption);
    Item.FilePath := tmpDocument.FilePath;
    fHandle := FileOpen(tmpDocument.FilePath + tmpDocument.FileName, fmOpenRead);
    if fHandle > - 1 then
      try
        Item.FileDate := DateTimeToStr(FileDateToDateTime(FileGetDate(fHandle)));
      finally
        FileClose(fHandle);
      end;
  end;
}
begin

  FJavaList.Clear;
  FHTMLList.Clear;
  FTextList.Clear;
  FOtherList.Clear;

  if FProject = nil then
    exit;

  FProject.GetDocList(FJavaList, dtJava);
  for i := 0 to FJavaList.Count -1 do
    FJavaList.Objects[i] := FBmpList[1];
  FProject.GetDocList(FHtmlList, dtHtml);
  for i := 0 to FHtmlList.Count -1 do
    FHtmlList.Objects[i] := FBmpList[2];
  FProject.GetDocList(FTextList, dtText);
  for i := 0 to FTextList.Count -1 do
    FTextList.Objects[i] := FBmpList[3];
  FProject.GetDocList(FOtherList, dtOther);
  for i := 0 to FOtherList.Count -1 do
    FOtherList.Objects[i] := FBmpList[4];
end;

procedure TdlgOptions.setProject(NewProject: TProject);
begin
  FLoadingProject := True;
  try
  if FProject <> nil then
    FProject.OnDocNameChange := nil;
  FProject := NewProject;
  edtProjectName.Enabled := (FProject <> nil);
  edtClassPath.Enabled := (FProject <> nil);
  edtDirectory.Enabled := (FProject <> nil);
  chkShowWarn.Enabled := (FProject <> nil);
  chkOptimize.Enabled := (FProject <> nil);
  chkVerbose.Enabled := (FProject <> nil);
  chkDebugInfo.Enabled := (FProject <> nil);
  if FProject = nil then begin
    edtProjectName.Text := '';
    edtClassPath.Text := '';
    edtDirectory.Text := '';
    chkShowWarn.Checked := False;
    chkOptimize.Checked := False;
    chkVerbose.Checked := False;
    chkDebugInfo.Checked := False;
    exit;
  end
  else begin
    FProject.OnDocNameChange := DocNameChange;
    edtProjectName.Text := FProject.Name;
    {Get Compiler Options}
    edtClassPath.Text := FProject.ClassPath;
    edtDirectory.Text := FProject.OutputPath;
    chkShowWarn.Checked := (pos('-nowarn',FProject.CompileOptions) = 0);
    chkOptimize.Checked := (pos('-O',FProject.CompileOptions) > 0);
    chkVerbose.Checked := (pos('-verbose',FProject.CompileOptions) > 0);
    chkDebugInfo.Checked := (pos('-g',FProject.CompileOptions) > 0);
  end;
  {get Documents}
  DisplayDocs;
  finally
    FLoadingProject := False;
  end;
end;

procedure TdlgOptions.DocNameChange(Sender: TObject);
begin
  DisplayDocs;
end;

{put list of docs into listbox}
procedure TdlgOptions.DisplayDocs;

  procedure LoadJava;
  var
    i: integer;
  begin
    for i := 0 to FJavaList.Count - 1 do begin
      lstDocuments.Items.AddObject(FJavaList[i],FJavaList.Objects[i])
    end;
  end;

  procedure LoadHtml;
  var
    i: integer;
  begin
    for i := 0 to FHtmlList.Count - 1 do begin
      lstDocuments.Items.AddObject(FHtmlList[i],FHtmlList.Objects[i])
{      LoadFileInfo(NewItem);}
    end;
  end;

  procedure LoadText;
  var
    i: integer;
  begin
    for i := 0 to FTextList.Count - 1 do begin
      lstDocuments.Items.AddObject(FTextList[i],FTextList.Objects[i])
{      LoadFileInfo(NewItem);}
    end;
  end;

  procedure LoadOther;
  var
    i: integer;
  begin
    for i := 0 to FOtherList.Count - 1 do begin
      lstDocuments.Items.AddObject(FOtherList[i],FOtherList.Objects[i])
{      LoadFileInfo(NewItem);}
    end;
  end;

begin

  lstDocuments.Items.Clear;
  InitDocLists;
  case tsDisplay.TabIndex of
    0: begin
       {All}
       LoadJava;
       LoadHtml;
       LoadText;
       LoadOther;
       end;
    1: LoadJava;
    2: LoadHtml;
    3: LoadText;
    4: LoadOther;
  end;
  lstDocuments.Refresh;

end;

procedure TdlgOptions.FormCreate(Sender: TObject);
var
  tmpBmp: TBitmap;
begin
  {Load the Bitmaps}
  FLoadingProject := False;

  FBmpList := TList.Create;
  tmpBmp := TBitmap.Create;
  tmpBmp.LoadFromResourceName(Hinstance, 'JE_BMPJAVA1');
  FBmpList.Add(tmpBmp);
  tmpBmp := TBitmap.Create;
  tmpBmp.LoadFromResourceName(HInstance, 'JE_BMPJAVA2');
  FBmpList.Add(tmpBmp);
  tmpBmp := TBitmap.Create;
  tmpBmp.LoadFromResourceName(HInstance, 'JE_BMPHTML');
  FBmpList.Add(tmpBmp);
  tmpBmp := TBitmap.Create;
  tmpBmp.LoadFromResourceName(HInstance, 'JE_BMPTEXT');
  FBmpList.Add(tmpBmp);
  tmpBmp := TBitmap.Create;
  tmpBmp.LoadFromResourceName(HInstance, 'JE_BMPOTHER');
  FBmpList.Add(tmpBmp);

  FJavaList := TStringList.Create;
  FHTMLList := TStringList.Create;
  FTextList := TStringList.Create;
  FOtherList := TStringList.Create;
  edtClassPath.Text := '';
  edtDirectory.Text := '';
  chkShowWarn.Checked := False;
  chkOptimize.Checked := False;
  chkVerbose.Checked := False;
  chkDebugInfo.Checked := False;
end;

procedure TdlgOptions.cmdOKClick(Sender: TObject);
begin
  SaveOptions;
  close;
end;

procedure TdlgOptions.SaveOptions;
var
  NewOptions: String;
begin

  if (FProject = nil) or FLoadingProject then
    exit;

  FProject.Name := Trim(edtProjectName.Text);
  FProject.ClassPath := Trim(edtClassPath.Text);
  FProject.OutputPath := Trim(edtDirectory.Text);

  NewOptions := '';
  if not chkShowWarn.Checked then
    NewOptions := ' -nowarn';
  if chkOptimize.Checked then
    NewOptions := NewOptions + ' -O';
  if chkVerbose.Checked then
    NewOptions := NewOptions + ' -verbose';
  if chkDebugInfo.Checked then
    NewOptions := NewOptions + ' -g';

  FProject.CompileOptions := NewOptions;

end;

procedure TdlgOptions.FormDestroy(Sender: TObject);
var
  i: integer;
  tmpBmp: TBitmap;
begin

  if Assigned(FBmpList) then begin
    for i := 0 to FBmpList.Count - 1 do begin
      tmpBmp := TBitmap(FBmpList[i]);
      if tmpBmp <> nil then begin
        tmpBmp.Free;
        FBmpList[i] := nil;
      end;
    end;
  end;
  FBmpList.Free;
  FBmpList := nil;
  FJavaList.Free;
  FHTMLList.Free;
  FTextList.Free;
  FOtherList.Free;
end;

procedure TdlgOptions.lstDocumentsFileDrop(Files: TStrings; X, Y: Integer);
var
  i: integer;
begin
  if FProject = nil then
    frmMain.NewProject(Self);
  for i := 0 to Files.Count - 1 do
    FProject.AddLoad(Files[i]);
  DisplayDocs;
end;

procedure TdlgOptions.lstDocumentsDblClick(Sender: TObject);
var
  i: integer;
  DocName: string;
begin

  for i := 0 to lstDocuments.Items.Count - 1 do begin
    if lstDocuments.Selected[i] then begin
      DocName := lstDocuments.Items[i];
      break;
    end;
  end;

  if length(DocName) >0 then begin
    frmMain.OpenProjectFile(DocName);
  end;

end;

function TdlgOptions.isSelection: Boolean;
var
  i: integer;
begin
  Result := False;

  for i := 0 to lstDocuments.Items.Count - 1 do begin
    if lstDocuments.Selected[i] then begin
      mnpCompile.Enabled := TypeFromExtension(ExtractFileExt(lstDocuments.Items[i])) = dtJava;
      mnpBrowse.Enabled := TypeFromExtension(ExtractFileExt(lstDocuments.Items[i])) = dtHtml;
      Result := True;
      break;
    end;
  end;

end;

procedure TdlgOptions.mnpDocumentPopup(Sender: TObject);
var
  i: integer;
  ItemSelected: Boolean;
begin

  mnpCompile.Enabled := False;
  mnpBrowse.Enabled := False;
  ItemSelected := isSelection;
  if ItemSelected then

    for i := 0 to lstDocuments.Items.Count - 1 do begin
      if lstDocuments.Selected[i] then begin
        mnpCompile.Enabled := TypeFromExtension(ExtractFileExt(lstDocuments.Items[i])) = dtJava;
        mnpBrowse.Enabled := TypeFromExtension(ExtractFileExt(lstDocuments.Items[i])) = dtHtml;
        break;
      end;
    end;

  mnpDocumentOpen.Enabled := ItemSelected;
  mnpDocumentDelete.Enabled := ItemSelected;
  mnpBrowse.Visible := mnpBrowse.Enabled;
  mnpCompile.Visible := mnpCompile.Enabled;

  mnpDocumentCut.Enabled := ItemSelected;
  mnpDocumentCopy.Enabled := ItemSelected;

  mnpDocumentPaste.Enabled :=  Clipboard.HasFormat(CF_TEXT)
                               and FileExists(Clipboard.AsText);

end;

procedure TdlgOptions.mnpDocumentOpenClick(Sender: TObject);
var
  i: integer;
begin

  for i := 0 to lstDocuments.Items.Count - 1 do begin
    if lstDocuments.Selected[i] then
      frmMain.OpenProjectFile(lstDocuments.Items[i]);
  end;

end;

procedure TdlgOptions.doDelete(Sender: TObject);
var
  i: integer;
begin

  for i := 0 to lstDocuments.Items.Count - 1 do begin
    if lstDocuments.Selected[i] then begin
      frmMain.CloseProjectFile(lstDocuments.Items[i]);
      FProject.Remove(lstDocuments.Items[i]);
    end;
  end;

  frmMain.UpdateMenuItems(nil);
  DisplayDocs;

end;

procedure TdlgOptions.doCopy(Sender: TObject);
var
  i: integer;
begin

  for i := 0 to lstDocuments.Items.Count - 1 do begin
    if lstDocuments.Selected[i] then begin
      frmMain.CloseProjectFile(lstDocuments.Items[i]);
      Clipboard.SetTextBuf(PChar(FProject.DocumentFileName(lstDocuments.Items[i])));
      break;
    end;
  end;

end;

procedure TdlgOptions.doCut(Sender: TObject);
var
  i: integer;
begin

  for i := 0 to lstDocuments.Items.Count - 1 do begin
    if lstDocuments.Selected[i] then begin
      frmMain.CloseProjectFile(lstDocuments.Items[i]);
      Clipboard.SetTextBuf(PChar(FProject.DocumentFileName(lstDocuments.Items[i])));
      FProject.Remove(lstDocuments.Items[i]);
      break;
    end;
  end;
  DisplayDocs;

end;

procedure TdlgOptions.doPaste(Sender: TObject);
begin
  if Clipboard.HasFormat(CF_TEXT) then
    if FileExists(Clipboard.AsText) then
      FProject.AddLoad(Clipboard.AsText);
  DisplayDocs;
end;

procedure TdlgOptions.lstDocumentsKeyUp(Sender: TObject; var Key: Word;
  Shift: TShiftState);
begin
  if Key = VK_DELETE then begin
    doDelete(Sender);
    DisplayDocs;
  end;
end;

procedure TdlgOptions.mnpAddClick(Sender: TObject);
begin
  {Change dialog extension for appropriate tab}
  if tsDisplay.TabIndex > 0 then
    frmMain.dlgOpen.FilterIndex := tsDisplay.TabIndex;
  frmMain.mnuProjectAddDocClick(self);
end;

procedure TdlgOptions.FormActivate(Sender: TObject);
begin
  DisplayDocs;
end;

{procedure rename
var
  tmpDocument: TDocument;
begin
  try
    tmpDocument := FProject.RenameDoc(Item.Caption, S);
  except
    begin
      S := Item.Caption;
      raise;
    end;
  end;
  S := tmpDocument.FileName;
end;
}
procedure TdlgOptions.mnpCompileClick(Sender: TObject);
var
  i: integer;
begin

  for i := 0 to lstDocuments.Items.Count - 1 do begin
    if lstDocuments.Selected[i] then
      Break;
  end;
  frmMain.Compile(lstDocuments.Items[i]);

end;

procedure TdlgOptions.mnpBrowseClick(Sender: TObject);
var
  i: integer;
begin

  for i := 0 to lstDocuments.Items.Count - 1 do begin
    if lstDocuments.Selected[i] then
      Break;
  end;
  frmMain.LoadInBrowser(lstDocuments.Items[i]);

end;

procedure TdlgOptions.FormClose(Sender: TObject; var Action: TCloseAction);
begin
  SaveOptions;
  Action := caFree;
  frmMain.OptionDlgOpen := False;
end;

procedure TdlgOptions.edtProjectNameChange(Sender: TObject);
begin
  SaveOptions;
end;

procedure TdlgOptions.lstDocumentsDrawItem(Control: TWinControl;
  Index: Integer; Rect: TRect; State: TOwnerDrawState);
var
  Bitmap: TBitmap;	{ temporary variable for the item's bitmap }
  Offset: Integer;	{ text offset width }
begin
  with (Control as TListBox).Canvas do	{ draw on the control canvas, not on the form }
  begin
    FillRect(Rect);	{ clear the rectangle }
    Offset := 2;	{ provide default offset }
    Bitmap := TBitmap((Control as TListBox).Items.Objects[Index]);	{ get the bitmap }
    if Bitmap <> nil then
    begin
      BrushCopy(Bounds(Rect.Left + 2, Rect.Top, Bitmap.Width, Bitmap.Height), Bitmap,
        Bounds(0, 0, Bitmap.Width, Bitmap.Height), clPurple);	{ render the bitmap }
      Offset := Bitmap.width + 6;	{ add four pixels between bitmap and text }
    end;
    TextOut(Rect.Left + Offset, Rect.Top, (Control as TListBox).Items[Index])	{ display the text }
  end;
end;


procedure TdlgOptions.tsDisplayClick(Sender: TObject);
begin
  DisplayDocs;
end;

end.
