{******************************************************************************}
{* Project:     Universal                                                     *}
{* Unit:        dosprocess                                                    *}
{* Description: Special Classes and utility procedures for Java Editor        *}
{*                                                                            *}
{* 1996,1997, Richard L. Chase, All Rights Reserved                          *}
{* Adapted from samples in MSDN, Copyright 1995, Microsoft Corporation       *}
{*                                                                            *}
{* Types:                                                                     *}
{*    EDosProcessError - Error raised by this unit                            *}
{*      readable.                                                             *}
{*                                                                            *}
{* Forms:                                                                     *}
{*    None                                                                    *}
{*                                                                            *}
{* Procedures:                                                                *}
{*    None                                                                    *}
{*                                                                            *}
{* Functions:                                                                 *}
{*    None                                                                    *}
{*                                                                            *}
{******************************************************************************}
unit dosprocess;

interface

uses SysUtils, Windows;

type
  EDosProcessError = class(Exception);

{CommandLine: Path to the app to run.
 tmpPath: location of output file.
 ShowWindow: show app window state
 ReturnStd: True for return Standard input, False for return Error data}
function RunDosProcess(CommandLine, tmpPath: string; ShowWindow, ReturnStd: Boolean): string;
function CreateChildProcess(CommandLine: string; ShowWindow: Boolean;
  hStdIn, hStdOut, hStdErr: THandle): boolean;

const
  DUPLICATE_CLOSE_SOURCE = $00000001;
  DUPLICATE_SAME_ACCESS  = $00000002;
  BUFSIZE                = 4096;


implementation

procedure WriteToPipe(hStdInWr, hStdInRd: THandle);
var
  dwRead, dwWritten: integer;
  chBuf: Array [0..BUFSIZE] of CHAR;
begin

  {Read from a file and write its contents to a pipe.}

  While True do begin
    if (not ReadFile(hStdInRd, chBuf, BUFSIZE, dwRead, nil) or (dwRead = 0)) then
      break;
    if (not WriteFile(hStdinWr, chBuf, dwRead, dwWritten, nil)) then
      break;
  end;

  {Close the pipe handle so the child stops reading.}
  if (not CloseHandle(hStdinWr)) then
    raise EDosProcessError.Create('Close pipe failed');

end;

function ReadFromPipe(hStdOutWr, hStdOutRd, hStdOutSave, hDataFile: THandle): string;
var
  dwRead, dwWritten: integer;
  chBuf: Array [0..BUFSIZE] of CHAR;
  ReadOK: Boolean;
  ReadMore: Boolean;
  NewStr: String;
begin

  {* Close the write end of the pipe before reading from the
   * read end of the pipe.}
  if (not CloseHandle(hStdoutWr)) then
    raise EDosProcessError.Create('Closing handle failed');
  {Read output from child, and write it to parent's STDOUT.}

  Repeat
    ReadOK := ReadFile(hStdOutRd, chBuf, BUFSIZE, dwRead, nil);
    NewStr := NewStr + string(chBuf);
    ReadMore := ReadOK and (dwRead <> 0);
    if ReadMore then
      {clear the read buffer by writing}
      ReadMore := WriteFile(hDataFile, chBuf, dwRead, dwWritten, nil);
    {
    if (not ReadOK) or (dwRead = 0) then
      break;
    if (not WriteFile(hStdoutSave, chBuf, dwRead, dwWritten, nil)) then
      break;
    }
  Until (Not Readmore);
  Result := NewStr;
end;

function RunDosProcess(CommandLine, tmpPath: string; ShowWindow, ReturnStd: Boolean): String;
var
  saAttr: TSECURITYATTRIBUTES;
  fSuccess: Boolean;
  hChildStdinRd, hChildStdinWr: THandle;
  hChildStdinWrDup, hChildStdoutRd: THandle;
  hChildStdoutWr: THandle;
  hSaveStdin, hSaveStdout,hSaveStdErr: THandle;
  hInputFile: THandle; {FOR USE IN WRITE PIPE}
  NewFile: String;
begin
  {Set the bInheritHandle flag so pipe handles are inherited.}
  saAttr.nLength := sizeof(TSecurityAttributes);
  saAttr.bInheritHandle := TRUE;
  saAttr.lpSecurityDescriptor := nil;

  {  * The steps for redirecting child's STDOUT:
     *     1.  Save current STDOUT, to be restored later.
     *     2.  Create anonymous pipe to be STDOUT for child.
     *     3.  Set STDOUT of parent to be write handle of pipe, so
     *         it is inherited by child.
  }

  {Save the handle to the current STDX.}
  hSaveStdin := GetStdHandle(STD_INPUT_HANDLE);
  hSaveStdout := GetStdHandle(STD_OUTPUT_HANDLE);
  hSaveStdErr := GetStdHandle(STD_ERROR_HANDLE);
{
  if ReturnStd then
    hSaveStdout := GetStdHandle(STD_OUTPUT_HANDLE)
  else hSaveStdout := GetStdHandle(STD_ERROR_HANDLE);
 }
  {Create a pipe for the child's STDOUT.}
  if (not CreatePipe(hChildStdoutRd, hChildStdoutWr, @saAttr, 0)) then
    raise EDosProcessError.Create('Stdout pipe creation failed');

  {Set a write handle to the pipe to be STDOUT or STDERR.}
  if ReturnStd then begin
    if (not SetStdHandle(STD_OUTPUT_HANDLE, hChildStdoutWr)) then
       raise EDosProcessError.Create('Redirecting STDOUT failed');
  end
  else begin
    if (not SetStdHandle(STD_ERROR_HANDLE, hChildStdoutWr)) then
       raise EDosProcessError.Create('Redirecting STDERR failed');
  end;


  {  * The steps for redirecting child's STDIN:
     *     1.  Save current STDIN, to be restored later.
     *     2.  Create anonymous pipe to be STDIN for child.
     *     3.  Set STDIN of parent to be read handle of pipe, so
     *         it is inherited by child.
     *     4.  Create a noninheritable duplicate of write handle,
     *         and close the inheritable write handle.
  }

  {Create a pipe for the child's STDIN.}
  if (not CreatePipe(hChildStdinRd, hChildStdinWr, @saAttr, 0)) then
    raise EDosProcessError.Create('Stdin pipe creation failed');

  {Set a read handle to the pipe to be STDIN.}
  if (not SetStdHandle(STD_INPUT_HANDLE, hChildStdinRd)) then
    raise EDosProcessError.Create('Redirecting Stdin failed');

  {Duplicate the write handle to the pipe so it is not inherited.}
  fSuccess := DuplicateHandle(GetCurrentProcess(),
                              hChildStdinWr,
                              GetCurrentProcess(),
                              @hChildStdinWrDup,
                              0,
                              FALSE,       {not inherited}
                              DUPLICATE_SAME_ACCESS);
  if (not fSuccess) then
    raise EDosProcessError.Create('DuplicateHandle failed');

  CloseHandle(hChildStdinWr);
{
  if ReturnStd then begin
    if (not SetStdHandle(STD_OUTPUT_HANDLE, hChildStdoutWr)) then
       raise EDosProcessError.Create('Redirecting STDOUT failed');
  end
  else begin
    if (not SetStdHandle(STD_ERROR_HANDLE, hChildStdoutWr)) then
       raise EDosProcessError.Create('Redirecting STDERR failed');
  end;
}
 try
 {Now create the child process., using the new in/out handles}
   if (not CreateChildProcess(CommandLine, ShowWindow,hChildStdinRd,hChildStdoutWr,hChildStdoutWr)) then
     raise EDosProcessError.Create('Create process failed');
 finally
   {After process creation, restore the saved STDIN and STDOUT.}
   if (not SetStdHandle(STD_INPUT_HANDLE, hSaveStdin)) then
     raise EDosProcessError.Create('Re-redirecting Stdin failed');
   if (not SetStdHandle(STD_OUTPUT_HANDLE, hSaveStdout)) then
     raise EDosProcessError.Create('Re-redirecting Stdout failed');
   if (not SetStdHandle(STD_ERROR_HANDLE, hSaveStdErr)) then
     raise EDosProcessError.Create('Re-redirecting Stderr failed');
 end;
{
 if ReturnStd then begin
   if (not SetStdHandle(STD_OUTPUT_HANDLE, hSaveStdout)) then
     raise EDosProcessError.Create('Re-redirecting Stdout failed');
 end
 else begin
   if (not SetStdHandle(STD_ERROR_HANDLE, hSaveStdout)) then
     raise EDosProcessError.Create('Re-redirecting Stderr failed');
 end;
}
 {Create a file to store results}
 NewFile := tmpPath + 'xxxxxxxx.xxx';
 GetTempFileName(pChar(tmpPath), 'jed', 0, PChar(NewFile));
 hInputFile := FileCreate(PChar(NewFile));
{
  Result := CreateFile(PChar(FileName), GENERIC_READ or GENERIC_WRITE,
    0, nil, CREATE_ALWAYS, FILE_ATTRIBUTE_NORMAL, 0);
}
 if (hInputFile = -1) then
   raise EDosProcessError.Create('no input file');
 try
 {get the ouput}
   ReadFromPipe(hChildStdoutWr,hChildStdoutRd, hSaveStdout, hInputFile);
 finally
   FileClose(hInputFile);
 end;
 Result := NewFile;

  {Get a handle to the parent's input file.

    if (argc > 1)
        hInputFile = CreateFile(argv[1], GENERIC_READ, 0, NULL,
            OPEN_EXISTING, FILE_ATTRIBUTE_READONLY, NULL);
    else
        hInputFile = hSaveStdin;

    if (hInputFile == INVALID_HANDLE_VALUE)
        raise EDosProcessError.Create("no input file\n");

    /* Write to pipe that is the standard input for a child process. */

    WriteToPipe(hChildStdinWrDup, hInputFile);

    /* Read from pipe that is the standard output for child process. */

    ReadFromPipe();

    return 0;
  }
end;

function CreateChildProcess(CommandLine: string; ShowWindow: Boolean;
  hStdIn, hStdOut, hStdErr: THandle): boolean;
var
  StartInfo: TSTARTUPINFO;
  ProcInfo: TProcessInformation;
begin

  {Set up members of STARTUPINFO structure.}

  FillChar(StartInfo, SizeOf(TStartupInfo), 0);
  with StartInfo do
  begin
    cb := SizeOf(TStartupInfo);
{
    dwFlags := STARTF_USESHOWWINDOW or STARTF_FORCEONFEEDBACK;
}
    dwFlags := STARTF_USESHOWWINDOW or STARTF_FORCEONFEEDBACK or STARTF_USESTDHANDLES;
    hStdOutput := hStdout;
    hStdInput := hStdin;
    hStdError := hStdErr;
    if ShowWindow then
      wShowWindow := SW_SHOWNORMAL
    else wShowWindow := SW_SHOWMINNOACTIVE;
  end;

  {Create the child process.}
  result :=  CreateProcess(nil,                 {application name}
                       PChar(CommandLine),      {command line}
                       nil,          {process security attributes}
                       nil,          {primary thread security attributes}
                       TRUE,          {handles are inherited}
                       0,             {creation flags}
                       nil,          {use parent's environment}
                       nil,          {use parent's current directory}
                       StartInfo,  {STARTUPINFO pointer}
                       ProcInfo);  {receives PROCESS_INFORMATION}

end;

end.
