{******************************************************************************}
{* Project:     JavaEdit                                                      *}
{* Unit:        JavaEditDefs                                                  *}
{* Description: Special Classes and utility procedures for Java Editor        *}
{*                                                                            *}
{* 1996,1997, Richard L. Chase, All Rights Reserved                          *}
{*                                                                            *}
{* Types:                                                                     *}
{*    TDirection - Load or Save - could use boolean, but this is more         *}
{*      readable.                                                             *}
{*    TDocumentType - Ordinals of Document type - for ID's                    *}
{*    TDocFilter - Stores File filter and Extension information about a       *}
{*      document type                                                         *}
{*    TDocTypeList - String list that holds document types and their labels   *}
{*                                                                            *}
{* Forms:                                                                     *}
{*    None                                                                    *}
{*                                                                            *}
{* Procedures:                                                                *}
{*    None                                                                    *}
{*                                                                            *}
{* Functions:                                                                 *}
{*    None                                                                    *}
{*                                                                            *}
{******************************************************************************}
unit javaeditdefs;

interface

uses
  SysUtils,
  Windows,
  Classes,
  Registry,
  graphics,
  dialogs,
  dosprocess;

type

  TDocumentType = (dtJava, dtHtml, dtText, dtOther);
  TDirection = (dLoad, dSave);

  TDocFilter = class
  public
    Filter: string;
    extension: string;
  end;

  TDocTypeList = class(TStringList)
  private
    procedure Initialize;
  public
    constructor Create;
    destructor destroy; override;
    procedure Clear; override;
  end;

  TProject = class;

  {onNameChange Event}
  TNameChangeEvent = procedure(Sender: TObject; NewName: string) of object;

  TDocument = class(TPersistent)
  private
    FDocType: TDocumentType;
    FFileName: string;
    FFilePath: string;
    FDocTitle: string;
    FNewDoc: Boolean;
    FDirty: Boolean;
    FProject: TProject;
    FOnNameChange: TNameChangeEvent;
    FisOpen: Boolean;
    procedure setFileName(NewName: string);
  public
{    constructor Create(AOwner: TComponent); override;}
    constructor Create;
    constructor CreateNew(ID: integer; DocType: TDocumentType); virtual;
    constructor CreateOpen(FileName: String); virtual;
    destructor Destroy; override;
    procedure RetrieveContents(Contents: TStrings);
    procedure Load(FileName: string);
    procedure Save(SourceText: TStrings);
    procedure Clear;
    function Compile(SourceText: TStrings): string;
    property Dirty: Boolean read FDirty write FDirty;
    property DocType: TDocumentType read FDocType write FDocType;
    property FileName: string read FFileName write setFileName;
    property FilePath: string read FFilePath write FFilePath;
    property Project: TProject read FProject;
    property Title: string read FDocTitle;
    Property isNew: Boolean read FNewDoc;
    Property isOpen: Boolean read FisOpen write FisOpen;
    property OnNameChange: TNameChangeEvent read FOnNameChange write FOnNameChange;
  end;

  TDocumentList = class(TStringList)
  private
  protected
    function GetDocument(Index: Integer): TDocument; virtual;
    procedure PutDocument(Index: Integer; AProperty: TDocument); virtual;
    procedure PutObject(Index: Integer; AObject: TObject); override;
  public
    constructor Create;
    destructor destroy; override;
{    function Add(const S: string): Integer; override;}
    function AddObject(const S: string; AObject: TObject): Integer; override;
    procedure InsertObject(Index: Integer; const S: string;
      AObject: TObject);
    procedure Clear; override;
    procedure Delete(Index: Integer); override;
{    procedure Insert(Index: Integer; const S: string); override;}
    property Documents[Index: Integer]: TDocument read GetDocument write PutDocument;
  end;

  TProject = class(TComponent)
  private
    fName: string;
    FFileName: string;
    FCompileOptions: string;
    FClassPath: string;
    FOutputPath: string;
    FAppClass: string;
    FMemberFiles: TDocumentList;
    FNewProject: Boolean;
    FDirty: Boolean;
    FOnDocNameChange: TNotifyEvent;
    function getDocCount: integer;
  protected
    procedure setCompileOrder(NewList: TStrings);
    procedure setAppClass(ClassName: string);
    procedure setClassPath(Value: string);
    procedure setCompileOptions(Value: string);
    procedure setFileName(Value: string);
    procedure SetProjectName(Value: string);
    procedure setOutputPath(Value: string);
    function getDirty: Boolean;
  public
    constructor Create(AOwner: TComponent); override;
    constructor CreateOpen(AOwner: TComponent; ProjFile: string); virtual;
    destructor destroy; override;
    function InProject(DocName: string): boolean;
    function DocumentFileName(DocName: string): string;
    procedure GetDocList(DocList: TStrings; DocType: TDocumentType);
    function GetDocIndex(DocName: string): integer;
    function GetDocName(Index: integer): string;
    procedure Add(Doc: TDocument);
    procedure AddLoad(FileName: string);
    procedure Remove(DocName: String);
    function RenameDoc(OldName: string; var NewName: string): TDocument;
    function Retrieve(DocName: String): TDocument;
    procedure Save;
    property AppClass: string read FAppClass write setAppClass;
    property CompileOptions: string read FCompileOptions write setCompileOptions;
    property CompileOrder: TStrings write SetCompileOrder;
    property ClassPath: string read FClassPath write setClassPath;
    property DocCount: integer read getDocCount;
    property FileName: string read FFileName write SetFileName;
    property IsDirty: Boolean read getDirty;
    property IsNew: Boolean read FNewProject;
    property Name: string read FName write SetProjectName;
    property OutputPath: string read FOutputPath write SetOutputPath;
    property OnDocNameChange: TNotifyEvent read FOnDocNameChange write FOnDocNameChange;
  end;

procedure OpenRegistryAppKey(var Registry: TRegistry);
function ExtensionFromType(DocType: TDocumentType): string;
function TypeFromExtension(Extension: string): TDocumentType;

const
  EXE_JAVAC                = 'javac.exe';  {*The Java compiler*}
  EXE_JAVA                 = 'java.exe';
  EXT_PROJECT              = '.jpr'; {Extension for Java projects}

  {* Ini/Registry keys/values *}

  KEY_CATEGORY                 = 'Software';
  KEY_COMPANY                  = 'dChase';
  KEY_PRODUCT                  = 'Java Editor';
    KEY_PRODUCT_DISPLAY        = 'Display';
      KEY_PRODUCT_DISPLAY_FONT = 'Font';
        INI_ITEM_FONT_NAME     = 'FontName';
        INI_ITEM_FONT_SIZE     = 'FontSize';
        INI_ITEM_FONT_COLOR    = 'FontColor';
      INI_ITEM_WIN95           = 'Win95Controls';
      INI_ITEM_LASTX           = 'LastX';
      INI_ITEM_LASTY           = 'LastY';
      INI_ITEM_LASTWINDOWSTATE = 'LastWindowState';
    KEY_PRODUCT_EDITOR         = 'Editor';
      INI_ITEM_TABSET        = 'TabSet';
      INI_ITEM_COLOR         = 'Color';
      INI_ITEM_AUTOINDENT    = 'AutoIndent';
      INI_ITEM_LASTFILETYPE  = 'LastFileType';
      INI_ITEM_DOCTABS       = 'ShowDocTabs';
    KEY_PRODUCT_JAVAC          = 'JavaCompiler';
      INI_ITEM_OPTIONS        = 'Options';
      INI_ITEM_CLASSPATH      = 'Classpath';
    KEY_PRODUCT_BROWSER        = 'WebBrowser';
      INI_ITEM_USEDDE         = 'UseDDE';
    KEY_PRODUCT_FILES          = 'LastFiles';
      INI_ITEM_FILES_PROJ      = 'Projects';
      INI_ITEM_FILES_DOC      = 'Documents';

  JPR_NAMEVERLAST    = 'Dick Chase''' + 's Java Editor 2.01';
  JPR_NAMEVER    = 'Dick Chase''' + 's Java(TM) Editor 2.08a';
  JPR_FILEVER    = '2';
  JPR_PROJECT    = 'Project';
  JPR_COMPILE    = 'CompileOptions';
  JPR_CLASSPATH  = 'ClassPath';
  JPR_OUTPATH    = 'OutputPath';
  JPR_FILES      = 'Files';

  VER_NUMBER    = '2.08a';
  VER_COPYRIGHT = 'Copyright  1996, Richard L. Chase';
  VER_RIGHTS    = 'All rights reserved';
  VER_LICENSE   = '';
  VER_SUPPORT   = 'Send questions or comments to dchase@tiac.net';

  MAXFILELIST   = 5;

var
  PATH_JAVAC: string;
  PATH_JAVA: string;
  PATH_BROWSER: string;
  JAVAC_OPTIONS: string;
  JAVAC_CLASSPATH: string;
  JAVAC_DIRECTORY: string;
  NETSCAPEWINDOW: string;
  NETSCAPEOPEN: Boolean;
  USEDDE: Boolean;
  USEWIN95: Boolean;
  TABSET: Shortint;
  AUTOINDENT: Boolean;
  EDITCOLOR: TColor;
  DOCTABS: Boolean;

implementation

{******************************************************************************}
{**                                                                          **}
{**                     General Procedures and Functions                     **}
{**                                                                          **}
{******************************************************************************}

procedure OpenRegistryAppKey(var Registry: TRegistry);
begin
  with Registry do begin
    closeKey;
    OpenKey(KEY_CATEGORY, True);
    OpenKey(KEY_COMPANY, True);
    OpenKey(KEY_PRODUCT, True);
  end;
end;

function ExtensionFromType(DocType: TDocumentType): string;
begin
  Result := '';
  case DocType of
    dtJava: Result := 'java';
    dtHtml: Result := 'html';
    dtText: Result := 'txt';
  end;
end;

function TypeFromExtension(Extension: string): TDocumentType;
var
  NewExt: string;
begin
  NewExt := UpperCase(Extension);
  if (NewExt = '.JAVA') or (NewExt = 'JAVA') then
    Result := dtJava
  else if (NewExt = 'HTM') or (NewExt = 'HTML')
        or (NewExt = '.HTM') or (NewExt = '.HTML') then
    Result := dtHtml
  else if (NewExt = 'TXT') or (NewExt = '.TXT') then
    Result := dtText
  else  Result := dtOther;
end;

{******************************************************************************}
{**                                                                          **}
{**                              TDocTypeList                                **}
{**                                                                          **}
{******************************************************************************}
constructor TDocTypeList.Create;
begin
  inherited Create;
  Initialize;
end;

destructor TDocTypeList.Destroy;
begin
  Clear;
  inherited Destroy;
end;

procedure TDocTypeList.Clear;
var i: integer;
begin
  for i := 0 to Count - 1 do
    if Objects[i] <> nil then
      Objects[i].Free;
  inherited Clear;
end;

procedure TDocTypeList.Initialize;
var
  tmpFilter: TDocFilter;
begin

  Clear;

  tmpFilter := TDocFilter.Create;
  tmpFilter.Filter := 'Java files (*.java)|*.java';
  tmpFilter.extension := 'java';
  AddObject('Java',tmpFilter);

  tmpFilter := TDocFilter.Create;
  tmpFilter.Filter := 'Html files (*.htm, *.html)|*.htm;*.html';
  tmpFilter.extension := 'html';
  AddObject('Html', tmpFilter);

  tmpFilter := TDocFilter.Create;
  tmpFilter.Filter := 'Text files (*.txt)|*.txt';
  tmpFilter.extension := 'txt';
  AddObject('Text', tmpFilter);

  tmpFilter := TDocFilter.Create;
  tmpFilter.Filter := 'All files (*.*)|*.*';
  tmpFilter.extension := '';
  AddObject('Other', tmpFilter);

end;

{******************************************************************************}
{**                                                                          **}
{**                              TDocument                                   **}
{**                                                                          **}
{******************************************************************************}
constructor TDocument.Create;
begin
  inherited Create;
{  if AOwner is TProject then
    FProject := (AOwner as TProject)
  else
}
  FProject := nil;
  FNewDoc := True;
  FDocType := dtJava;
  FFileName := '';
  FFilePath := '';
  FDirty := False;
  FOnNameChange := nil;
end;

constructor TDocument.CreateNew(ID: integer; DocType: TDocumentType);
begin
  Create;
  FFilePath := GetCurrentDir;
  FDocType := DocType;
  FDocTitle := 'New_' + IntToStr(ID) + '.' + ExtensionFromType(FDocType);
  FDirty := True;
  FOnNameChange := nil;
end;

constructor TDocument.CreateOpen(FileName: String);
begin
  if not FileExists(FileName) then
    raise Exception.Create('Can''' + 't find ''' + FileName + '''');
  Create;
  FOnNameChange := nil;
  Load(FileName);
end;

destructor TDocument.destroy;
begin
  inherited Destroy;
end;

procedure TDocument.setFileName(NewName: string);
var
  tmpFile: TStringList;
  tmpName: string;
begin
  if NewName = FFileName then
    Exit;
  if Length(NewName) = 0 then
    exit;

  tmpName := copy(NewName,1, Length(NewName) - Length(ExtractFileExt(NewName)));
  tmpName := tmpName + '.' + ExtensionFromType(FDocType);

  {Create new file and delete old}
  tmpFile := TStringList.Create;
  try
    tmpFile.LoadFromFile(FFilePath + FFileName);
    tmpFile.SaveToFile(FFilePath + tmpName);
  finally
    tmpFile.Free;
  end;
  DeleteFile(PChar(FFilePath + FFileName));

  {Change the name and notify owner}
  FFileName := tmpName;
  if Assigned(FOnNameChange) then
    FOnNameChange(Self,FFileName);

end;

procedure TDocument.Clear;
begin
  FDocType := dtJava;
  FFileName := '';
  FFilePath := '';
end;

procedure TDocument.Load(FileName: string);
var
  i: integer;
begin
  i := -1;
  if not FileExists(FileName) then
    raise Exception.Create('Can''' + 't find ''' + FileName + '''');
  {get the index of doc in project}
  if FProject <> nil then
    i := FProject.GetDocIndex(FFileName);
  Clear;
  FFileName := ExtractFileName(FileName);
  FFilePath := ExtractFilePath(FileName);
  FDocType := TypeFromExtension(ExtractFileExt(FFileName));
  FDocTitle := FFileName;
  FNewDoc := False;
  FDirty := False;
  if Assigned(FOnNameChange) then
    FOnNameChange(Self,FFileName);
  if (i >= 0) then begin
    FProject.FMemberFiles[i] := FFileName;
    if assigned(FProject.FOnDocNameChange) then
      FProject.FOnDocNameChange(FProject);
  end;

end;

procedure TDocument.RetrieveContents(Contents: TStrings);
begin
  Contents.LoadFromFile(FFilePath + FFileName);
end;

procedure TDocument.Save(SourceText: TStrings);
begin
  SourceText.SaveToFile(FFilePath + FFileName);
  FNewDoc := False;
  FDirty := False;
end;

function TDocument.Compile(SourceText: TStrings): string;
var
  CommandLine: string;
  ErrorFile: string;
  tmpContents, ErrMsg: TStringList;
  tmpData: TStringList;
  isOldJava: Boolean;
  FileAttribute: integer;
  isReadOnly: boolean;
begin

  Result := '';

  ErrMsg := TStringList.Create;
  try
    if FDocType <> dtJava then begin
      GetTempFileName(pChar(FFilePath), 'jed', 0, PChar(ErrorFile));
      ErrMsg.Add('''' + FFileName + ''' is not a java document.');
    end
    else if FNewDoc then  begin
      GetTempFileName(pChar(FFilePath), 'jed', 0, PChar(ErrorFile));
      ErrMsg.Add('''' + FFileName + ''' has not been saved yet.');
    end;
  finally
    if ErrMsg.Count > 0 then begin
      ErrMsg.SavetoFile(ErrorFile);
      ErrMsg.Free;
      Result := ErrorFile;
    end;
  end;

  if length(Result) > 0 then
    exit;

  FileAttribute := FileGetAttr(FFilePath + FFileName);
  isReadOnly := (faReadOnly and FileAttribute > 0);

  tmpContents := TStringList.Create;
  try
    if not isReadOnly then begin
      tmpContents.LoadFromFile(FFilePath + FFileName);
      SourceText.SaveToFile(FFilePath + FFileName);
    end;
    try
      CommandLine := '"' + PATH_JAVAC + '" '
                   + JAVAC_OPTIONS
                   + JAVAC_CLASSPATH
                   + JAVAC_DIRECTORY
                   + ' "' + FFilePath + FFileName + '"';

{Standard handles: in JDK 1.0.x, compiler uses STD_OUTPUT_HANDLE
                   in JDK 1.1.x, compiler uses STD_ERROR_HANDLE
}
      {get the version of the JDK}
{      isOldJava := True;}
      (* *)
      isOldJava := false;
      Result := RunDosProcess('"' + PATH_JAVA + '" -version',FFilePath,false,false);
      if Length(Result) > 0 then begin
        if FileExists(Result) then begin
          tmpData := TStringList.Create;
          try
            tmpData.LoadFromFile(Result);
            {java versions less than 1.1 are labeled "java version "1.0...}
            isOldJava := (pos('java version "1.0',tmpData[0]) > 0);
          finally
            tmpData.Free;
            DeleteFile(pchar(result));
          end;
        end;
      end;
(*      *)
      {if the version is 1.1 or better, run with stderr, else run with stdin}

      Result := RunDosProcess(CommandLine,FFilePath, False, isOldJava);
{
      Result := RunDosProcess(CommandLine,FFilePath, False);
}
    finally
      if not isReadOnly then tmpContents.SaveToFile(FFilePath + FFileName);
    end;
  finally
    tmpContents.Free;
  end;

end;

{******************************************************************************}
{**                                                                          **}
{**                            TDocumentList                                 **}
{**                                                                          **}
{******************************************************************************}
constructor TDocumentList.Create;
begin
  inherited Create;
end;

destructor TDocumentList.destroy;
begin
  Clear;
  inherited Destroy;
end;

function TDocumentList.GetDocument(Index: Integer): TDocument;
begin
  result := (getObject(Index) as TDocument);
end;

procedure TDocumentList.PutDocument(Index: Integer; AProperty: TDocument);
begin
{
  if objects[index] <> nil then begin
    objects[index].free;
    objects[index] := nil;
  end;
}
  putObject(Index, AProperty);
end;

procedure TDocumentList.PutObject(Index: Integer; AObject: TObject);
begin
  if (AObject <> nil) and (not(AObject is TDocument)) then
    raise Exception.Create('not a TDocument');
  inherited PutObject(Index, AObject);
end;

{
function TDocumentList.Add(const S: string): Integer;
var
  tmpProperty: TDocument;

begin
  Result := inherited add(S);
  tmpProperty := TDocument.Create(nil);
  putObject(Result, tmpProperty);
end;
}
function TDocumentList.AddObject(const S: string; AObject: TObject): Integer;
begin
  if not(AObject is TDocument) then
    raise Exception.Create('not a TDocument');
  result := inherited AddObject(S,AObject);
end;

procedure TDocumentList.InsertObject(Index: Integer; const S: string;
  AObject: TObject);
begin
  if not(AObject is TDocument) then
    raise Exception.Create('not a TDocument');
  inherited Insert(Index, S);
  PutObject(Index, AObject);
end;

procedure TDocumentList.Clear;
{var
  i: integer;}
begin
{  for i := 0 to count -1 do
  if Objects[i] <> nil then begin
    if (Objects[i] as TDocument).Owner = nil then
      (Objects[i] as TDocument).Free;
    Objects[i] := nil;
  end;
}
  inherited Clear;
end;

procedure TDocumentList.Delete(Index: Integer);
begin

  if Objects[Index] <> nil then begin
    Objects[Index] := nil;
  end;

  inherited Delete(Index);
  
end;
{
procedure TDocumentList.Insert(Index: Integer; const S: string);
var
  tmpProperty: TDocument;
begin

  inherited Insert(Index, S);
  tmpProperty := TDocument.Create(nil);
  putObject(Index, tmpProperty);

end;
}
{******************************************************************************}
{**                                                                          **}
{**                                 TProject                                 **}
{**                                                                          **}
{******************************************************************************}
constructor TProject.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  fName :='New Project';
  FFileName :='';
  FCompileOptions :='';
  FClassPath :='';
  FOutputPath :='';
  FMemberFiles := TDocumentList.Create;
  FNewProject := True;
  FDirty := False;
end;

constructor TProject.CreateOpen(AOwner: TComponent; ProjFile: string);
var
  tmpList: TStringList;
  i, next: integer;
begin
  Create(AOwner);
  if not FileExists(ProjFile) then
    raise Exception.Create('Can''' + 't find ''' + ProjFile + '''');

  FFileName := ProjFile;

  FNewProject := False;

  tmpList := TStringList.Create;
  try

    tmpList.LoadFromFile(ProjFile);

    if (tmpList.Count <2) or (tmpList[1] <> JPR_FILEVER) then
      raise Exception.Create('''' + ProjFile + ''' is not a JavaEdit Project or is from an earlier version of JavaEdit.');

    if (tmpList[1] <> JPR_FILEVER) and (tmpList[0] = JPR_NAMEVERLAST) then
    {it's an earlier version with compatible file structure}
      tmpList.Insert(1,JPR_FILEVER);

    Next := 1;
    {get project Name]}
    for i := Next to tmpList.Count - 1 do begin
      inc(Next);
      if Trim(tmpList[i]) = JPR_PROJECT then begin
        FName := Trim(tmpList[Next]);
        inc(Next);
        break;
      end;
    end;

    {get project options]}
    for i := Next to tmpList.Count - 1 do begin
      inc(Next);
      if Trim(tmpList[i]) = JPR_COMPILE then begin
        FCompileOptions := Trim(tmpList[i + 1]);
        inc(Next);
        break;
      end;
    end;

    {get Class Path]}
    for i := Next to tmpList.Count - 1 do begin
      inc(Next);
      if Trim(tmpList[i]) = JPR_CLASSPATH then begin
        FClassPath := Trim(tmpList[i + 1]);
        inc(Next);
        break;
      end;
    end;

    {get Output Path]}
    for i := Next to tmpList.Count - 1 do begin
      inc(Next);
      if Trim(tmpList[i]) = JPR_OUTPATH then begin
        FOutputPath := Trim(tmpList[i + 1]);
        inc(Next);
        break;
      end;
    end;

    {get files}
    for i := Next to tmpList.Count - 1 do begin
      inc(Next);
      if Trim(tmpList[i]) = JPR_FILES then begin
        break;
      end;
    end;

    for i := Next to tmpList.Count - 1 do begin
      if length(Trim(tmpList[i])) > 0 then
        if fileExists(tmpList[i]) then begin
          AddLoad(tmpList[i]);
        end;
    end;

    FDirty := False;

    SetCurrentDir(ExtractFileDir(FileName));

  finally
    tmpList.Free;
  end;
end;

destructor TProject.destroy;
var
  i: integer;
  tmpDoc: TDocument;
begin
  for i := 0 to FMemberFiles.Count - 1 do begin
    if FMemberFiles.documents[i] <> nil then begin
      tmpDoc := FMemberFiles.documents[i];
      tmpDoc.Free;
    end;
  end;
  FMemberFiles.Free;
  inherited Destroy;
end;

procedure TProject.GetDocList(DocList: TStrings; DocType: TDocumentType);
var
  i: integer;
begin
  DocList.Clear;
  for i := 0 to FMemberFiles.Count - 1 do begin
    if FMemberFiles.documents[i] <> nil then begin
      if FMemberFiles.Documents[i].DocType = DocType then
        DocList.Add(FMemberFiles[i]);
    end;
  end;
end;

procedure TProject.Save;
var
  tmpList: TStringList;
  i: integer;
  tmpDocument: TDocument;
begin
  tmpList := TStringList.Create;
  try
    tmpList.Add(JPR_NAMEVER);
    tmpList.Add(JPR_FILEVER);
    tmpList.Add(JPR_PROJECT);
    tmpList.Add('  ' + FName);
    tmpList.Add('');
    tmpList.Add(JPR_COMPILE);
    tmpList.Add('  ' + FCompileOptions);
    tmpList.Add('');
    tmpList.Add(JPR_CLASSPATH);
    tmpList.Add('  ' + FClassPath);
    tmpList.Add('');
    tmpList.Add(JPR_OUTPATH);
    tmpList.Add('  ' + FOutputPath);
    tmpList.Add('');
    tmpList.Add(JPR_FILES);

    for i := 0 to FMemberFiles.Count - 1 do begin
      if FMemberFiles.documents[i] <> nil then begin
        tmpDocument := FMemberFiles.Documents[i];
        tmpList.Add(tmpDocument.FilePath + tmpDocument.FileName);
      end;
    end;

    tmpList.SaveToFile(FFileName);

    FNewProject := False;
    FDirty := False;

    SetCurrentDir(ExtractFileDir(FileName));

  finally
    tmpList.Free;
  end;
end;

procedure TProject.setCompileOrder(NewList: TStrings);
var
  i, index: integer;
begin
  for i := NewList.Count - 1 to 0 do begin
    index := FMemberFiles.IndexOf(NewList[i]);
    if index >= 0 then
      FMemberFiles.Move(index,0);
  end;
  FDirty := True;
end;

procedure TProject.Add(Doc: TDocument);
begin
  FMemberFiles.AddObject(Doc.FileName,Doc);
  Doc.FProject := Self;
  FDirty := True;
end;

procedure TProject.AddLoad(FileName: string);
var
  tmpDocument: TDocument;
begin

  if not FileExists(FileName) then
    exit;
  {make sure file isn't already included}
  if FMemberFiles.IndexOf(ExtractFileName(FileName)) >=0 then
    exit;

  tmpDocument := TDocument.CreateOpen(FileName);
  Add(tmpDocument);

end;

procedure TProject.Remove(DocName: String);
var
  i: integer;
  tmpDocument: TDocument;
begin
  for i := 0 to FMemberFiles.Count - 1 do begin
    if FMemberFiles[i] = DocName then begin
      tmpDocument := FMemberFiles.Documents[i];
      FMemberFiles.Delete(i);
      if not tmpDocument.isOpen then
        tmpDocument.Free
      else tmpDocument.FProject := nil;
      break;
    end;
  end;

  FDirty := True;

end;

function TProject.GetDocIndex(DocName: string): Integer;
begin
  result := FMemberFiles.indexof(DocName);
end;

function TProject.GetDocName(Index: integer): string;
begin
  result := FMemberFiles[Index];
end;

function TProject.Retrieve(DocName: String): TDocument;
var
  i: integer;
begin
  Result := nil;
  for i := 0 to FMemberFiles.Count - 1 do begin
    if FMemberFiles[i] = DocName then begin
      Result := FMemberFiles.Documents[i];
      Break;
    end;
  end;
end;

procedure TProject.setAppClass(ClassName: string);
begin
  if FMemberFiles.indexof(ClassName) < 0 then
    exit;
  if FAppClass = ClassName then
    exit;
  FAppClass := ClassName;
  FDirty := True;
end;

procedure TProject.setClassPath(Value: string);
begin
  if FClassPath = Value then
    exit;
  FClassPath := Value;
  FDirty := True;
end;

procedure TProject.setCompileOptions(Value: string);
begin
  if FCompileOptions = Value then
    exit;
  FCompileOptions := CompileOptions;
  FDirty := True;
end;

procedure TProject.setFileName(Value: string);
begin
  if FFileName = Value then
    exit;
  FFileName := Value;
  FDirty := True;
end;

procedure TProject.setProjectName(Value: string);
begin
  if FName = Value then
    exit;
  FName := Value;
  FDirty := True;
end;

procedure TProject.setOutputPath(Value: string);
begin
  if FOutputPath = Value then
    exit;
  FOutputPath := Value;
  FDirty := True;
end;

function TProject.InProject(DocName: string): boolean;
begin
  Result := length(DocumentFileName(DocName)) > 0;
end;

function TProject.DocumentFileName(DocName: string): string;
var
  i: integer;
  tmpDocument: TDocument;
begin
  Result := '';
  for i := 0 to FMemberFiles.Count - 1 do begin
    if FMemberFiles[i] = DocName then begin
      if FMemberFiles.documents[i] <> nil then begin
        tmpDocument := FMemberFiles.Documents[i];
        Result := tmpDocument.FilePath + tmpDocument.FileName;
      end;
    end;
  end;
end;

function TProject.getDirty: Boolean;
var
  i: integer;
  tmpDocument: TDocument;
begin
  Result := FDirty;
  if Result then
    exit;
  for i := 0 to FMemberFiles.Count - 1 do begin
    if FMemberFiles.documents[i] <> nil then begin
      tmpDocument := FMemberFiles.Documents[i];
      Result := tmpDocument.Dirty;
      if Result then
        Break;
    end;
  end;
end;

function TProject.RenameDoc(OldName: string; var NewName: string): TDocument;
var
  tmpDocument: TDocument;
begin
  tmpDocument := Retrieve(OldName);
  tmpDocument.FileName := NewName;
  FDirty := True;
  Result := tmpDocument;
end;

function TProject.getDocCount: integer;
begin
  Result := FMemberFiles.Count;
end;

end.
