/**
 * Example 5: Chaining ValueModels
 *
 * This example shows how a model can be dependent on
 * another model.
 *
 * There are two components in the applet, both MVCTextFields.  If you
 * enter a number in the enterable text field, then two times that
 * value will be placed in the read-only text field.  This will example
 * demonstrates how to use the ValueAdaptor abstract class.
 */

import java.awt.*;
import com.bdnm.mvc.*;

public class Example5 extends java.applet.Applet {

    /**
     * This is the only method we really need.  It is
     * called when the applet is starting up.  This is
     * where we layout the panel.
     *
     * Note that we don't need to hold onto the model.  We can hook
     * everything together in this method.
     */
    public void init() {
        // Just give the examples a distinctive background
        setBackground(new Color(128,128,192));

        // First, create the widget, make it a Float.  Stick a label out
        // there, too.
        add(new Label("Base Value: "));
        MVCTextField baseValue = new MVCTextField(15);
        baseValue.setType(MVCTextField.TYPE_FLOAT);
        add(baseValue);

        // Now add the second text field, make it read-only and a Float.
        add(new Label("Times Two: "));
        MVCTextField timesTwo = new MVCTextField(15);
        timesTwo.setType(MVCTextField.TYPE_FLOAT);
        timesTwo.setEditable(false);
        add(timesTwo);

        // Here comes the fun.  Make the second text field's model be
        // a ValueAdaptor that we set up to be two times the first text
        // field's model.  Take a look at the non-public class below to
        // see how this is set up.
        timesTwo.setModel(new TimesTwoValueAdaptor(baseValue.getModel()));
        //timesTwo.setModel(baseValue.getModel());
    }

    /**
     * This is just a method to make this run standalone.
     */
    public static void main(String args[]) {
	    Frame f = new Frame("Example 5");
	    Example5 applet = new Example5();

	    applet.init();

	    f.add("Center", applet);
	    f.pack();
    	f.resize(f.preferredSize());
	    f.show();
    }
}

/**
 * This class holds onto a model and makes sure that
 * the value returned by getValue is two times the
 * reference model.  Use this as an example for any
 * general PluggableAdaptors.
 */
class TimesTwoValueAdaptor extends ValueAdaptor {
    /**
     * Construct an instance with a reference model.
     *
     * @param   aModel   a reference model
     */
    public TimesTwoValueAdaptor(ValueModel aModel) {
        super(aModel);
    }

    /**
     * The value coming from my model can be modified.  Return
     * the modified value that will be forwarded on.
     *
     * I will pass along two times my model's value
     *
     * @param   aValue  the value of my model
     * @return the value which I represent
     */
    protected Object valueFromModel(Object aValue) {
        try {
            Number refValue = (Number) aValue;
            return new Float(refValue.floatValue() * 2.0);
        } catch (ClassCastException e) {
            return new Float(0.0);
        }
    }

    /**
     * The value to set to my model can be modified.  Return the
     * modified value that my model will be set to.
     *
     * @param   aValue  the value to I should represent
     * @return the value which my model should be set to
     */
     protected Object valueToModel(Object aValue) {
        float numberValue;
        try {
            numberValue = ((Number) aValue).floatValue();
        } catch (ClassCastException e) {
            numberValue = 0.0f;
        }
        return new Float(0.5*numberValue);
     }

}