/**
 * Example 7: An MVCList of Players
 *
 * This example shows how the list model behind an MVCList
 * can work with any Object.
 *
 * There are four components in this application.  The MVCList on
 * the left is using the DefaultObjectPrinter which sends toString() to
 * each object.  The MVCList on the right is using an ObjectPrinter that
 * we created to display the names a little differently.  There are also
 * two Buttons for populating the list, which, by the way, both MVCList
 * components share.
 */

import java.awt.*;
import java.util.Vector;
import com.bdnm.mvc.*;

public class Example7 extends java.applet.Applet {

    /**
     * This the is model behind both lists which holds
     * onto both the list and the selections.
     */
    SelectionInList listModel;

    /**
     * Hold onto the instances of the buttons - I like
     * this model to figure out what button has been pressed.
     */
    Button bullsButton, magicButton;

    /**
     * Hold onto the two teams: the Bulls and Magic.
     */
    //static Team bulls = Team.bulls(), magic = Team.magic();

    /**
     * Layout the panel here and hold onto the model
     * behind the list.
     */
    public void init() {
        // Just give the examples a distinctive background
        setBackground(new Color(128,128,192));

        // Make sure we have a border layout for this applet.
        // This layout is simple enough that we can use it.
        setLayout(new BorderLayout());

        // Create one MVCList, add it to the left
        MVCList leftPlayerList = new MVCList();
        add("West",leftPlayerList);

        // Create the other MVCList, add it to the right
        MVCList rightPlayerList = new MVCList();
        add("East",rightPlayerList);

        // Now add the two buttons
        add("North", bullsButton = new Button("The Bulls"));
        add("South", magicButton = new Button("The Magic"));

        // Now just hold onto the list's model so we can
        // modify it later.
        listModel = leftPlayerList.getModel();

        // Now we'll give the right list the exact same model.
        // This will sync up both the lists and the selections.
        rightPlayerList.setModel(listModel);

        // Now setup the right-side list to use a special
        // ObjectPrinter - not the default.
        rightPlayerList.objectPrinter(new PlayerObjectPrinter());
    }

    /**
     * If one of our buttons was pressed, fill out the list.
     */
    public boolean action(Event evt, Object what) {
        if (evt.target == bullsButton) {
            populateListFromTeam(Team.bulls());
        } else if (evt.target == magicButton) {
            populateListFromTeam(Team.magic());
        } else {
            return false;
        }
        return true;
    }

    /**
     * Make the list have the Player objects from the given team.
     */
    private void populateListFromTeam(Team team) {
        Player[] players = team.players();
        Vector newList = new Vector(players.length);
        for (int i = 0; i < players.length; ++i) {
            newList.addElement(players[i]);
        }
        listModel.list(newList);
    }


    /**
     * This is just a method to make this run standalone.
     */
    public static void main(String args[]) {
	    Frame f = new Frame("Example 7");
	    Example7 applet = new Example7();

	    applet.init();

	    f.add("Center", applet);
	    f.pack();
    	f.resize(f.preferredSize());
	    f.show();
    }
}

/**
 * Here's a different way to represent Players.  This method is used
 * for the right-hand list.
 */
class PlayerObjectPrinter implements ObjectPrinter {

    /**
     * This is the only method I have to implement.
     */
    public String toString(Object obj) {
        Player aPlayer = (Player) obj;
        return aPlayer.lastName() + ", " + aPlayer.firstName();
    }
}