/**
 * Object Model: Player
 *
 * This is the business object model our application is focussed on.
 * It consists of teams and players.  This file holds the definition
 * of the Player object.
 */
import java.util.Date;

public class Player extends java.util.Observable {

    /**
     * A player has a last name and a first name.
     */
    String lastName, firstName;

    /**
     * A player has a birthdate, represented as a Date
     */
    Date birthdate;

    /**
     * We also shooting percentage for the player.
     */
    float shootingPercentage;

    /**
     * The minimum information you need to construct a player is
     * the first name, last name, and birthdate.  These are not
     * changeable after the player is created.
     */
    public Player(String lastName, String firstName, Date birthdate) {
        super();
        this.lastName = lastName;
        this.firstName = firstName;
        this.birthdate = birthdate;
    }

    /**
     * This is a convenience constructor to completely set up a player
     * at construction time.
     */
    public Player(String lastName, String firstName, Date birthdate, float shootingPercentage) {
        this(lastName, firstName, birthdate);
        this.shootingPercentage = shootingPercentage;
    }

    /**
     * The getter for the player's last name.  Note that there is no setter.
     * @return the last name of the player
     */
    public String lastName() {
        return lastName;
    }

    /**
     * The getter for the player's first name.  Note that there is no setter.
     * @return the first name of the player
     */
    public String firstName() {
        return firstName;
    }

    /**
     * The getter for the player's birthdate.  Note that there is no setter.
     * @return the birthdate of the player
     */
    public Date birthdate() {
        return birthdate;
    }

    /**
     * The getter for the shooting percentage.
     * @return the shooting percentage for this player
     */
    public float shootingPercentage() {
        return shootingPercentage;
    }

    /**
     * The setter for the shooting percentage.  Validate the argument so
     * it only takes values from 0.0 to 1.0.  Also announce the change.
     * @param   newPercentage   shooting percentage for the player (0.0 to 1.0)
     */
    public void shootingPercentage(float newPercentage) {
        if (newPercentage >= 0.0 && newPercentage <= 1.0) {
            shootingPercentage = newPercentage;
        }
        // We'll announce the change even if it didn't so things
        // get refreshed properly
        changed("shootingPercentage");
    }

    /**
     * Here's a sample function that calculates the age of the player.
     * @return the age of the player (in years, rounded down) today
     */
    public int age() {
        // I don't know if this is the best way to do this, but
        // it seems to work
        Date today = new Date();
        int birthYear = birthdate.getYear();
        int age = today.getYear() - birthYear;
        today.setYear(birthYear);
        return birthdate.after(today) ? age : age + 1;
    }

    /**
     * Create a resonable print representation: the player's first name,
     * a space, and the last name.
     * @return a String represenation of a player
     */
    public String toString() {
        return firstName + " " + lastName;
    }

    /**
     * This is just a convenience method that notifies any Observers
     * that we have changed.  The argument that I pass along is the
     * name of the attribute that changed.
     */
    protected void changed(String attributeName) {
        setChanged();
        notifyObservers(attributeName);
    }
}
