/**
 * Object Model: Team
 *
 * This is the business object model our application is focussed on.
 * It consists of teams and players.  This file holds the definition
 * of the Team object.
 */
import java.util.Date;

public class Team extends java.util.Observable {

    /**
     * A team has a name.  This is really the "key", or unique
     * identifier, for the object.
     */
    String name;

    /**
     * A team also has a record of the number of wins and losses
     * for the current season.
     */
    int wins, losses;

    /**
     * Finally, a team has players.  Store them in an array.
     */
    Player[] players;

    /**
     * In order to construct a team, we need to provide the name.
     */
    public Team(String teamName) {
        super();
        name = teamName;
    }

    /**
     * The getter for the team's name.  Note that there is no setter.
     * @return the name of the team
     */
    public String name() {
        return name;
    }

    /**
     * The getter for the number of wins.
     * @return the number of team wins this season
     */
    public int wins() {
        return wins;
    }

    /**
     * The setter for the number of wins.  Also announce the change.
     * @param   newWins the number of team wins this season
     */
    public void wins(int newWins) {
        wins = newWins;
        changed("wins");
    }

    /**
     * The getter for the number of losses.
     * @return the number of team losses this season
     */
    public int losses() {
        return losses;
    }

    /**
     * The setter for the number of losses.  Also announce the change.
     * @param   newLosses the number of team losses this season
     */
    public void losses(int newLosses) {
        losses = newLosses;
        changed("losses");
    }

    /**
     * The getter for the players.  Just return the entire array.
     * @return an array of players
     */
    public Player[] players() {
        return players;
    }

    /**
     * The setter for players.  Note that we're not getting cute and
     * allowing users of this class to add and delete individual
     * players.
     * @param   newPlayers  the players for the team
     */
    public void players(Player[] newPlayers) {
        players = newPlayers;
        changed("players");
    }

    /**
     * Return the team name as a default String representation.
     * @return the team name
     */
    public String toString() {
        return name;
    }

    /**
     * This is just a convenience method that notifies any Observers
     * that we have changed.  The argument that I pass along is the
     * name of the attribute that changed.
     */
    protected void changed(String attributeName) {
        setChanged();
        notifyObservers(attributeName);
    }

    // For both of the following methods, Date cannot represent a date
    // from before 1970.  Therefore, all players are 10 years older than
    // represented here.

    /**
     * This creates a Team object which represents the '95-'96 Chicago
     * Bulls.
     * @return a newly created Team representing the Bulls
     */
    static public Team bulls(){
        Team team = new Team("Chicago Bulls");
        team.wins(72);
        team.losses(10);
        Player[] players = new Player[5];
        players[0] = new Player("Pippen", "Scottie", new Date(75,8,25), 0.463f);
        players[1] = new Player("Rodman", "Dennis", new Date(71,4,13), 0.480f);
        players[2] = new Player("Longley", "Luc", new Date(79,0,19), 0.482f);
        players[3] = new Player("Harper", "Ron", new Date(74,0,20), 0.467f);
        players[4] = new Player("Jordan", "Micheal", new Date(73,1,17), 0.495f);
        team.players(players);
        return team;
    }

    /**
     * This creates a Team object which represents the '95-'96 Orlando
     * Magic.
     * @return a newly created Team representing the Magic
     */
    static public Team magic(){
        Team team = new Team("Orlando Magic");
        team.wins(60);
        team.losses(22);
        Player[] players = new Player[5];
        players[0] = new Player("Scott", "Dennis", new Date(78,8,5), 0.440f);
        players[1] = new Player("O'Neal", "Shaquille", new Date(82,2,6), 0.573f);
        players[2] = new Player("Hardaway", "Anfernee", new Date(82,6,18), 0.513f);
        players[3] = new Player("Andersen", "Nick", new Date(78,0,20), 0.442f);
        players[4] = new Player("Grant", "Horace", new Date(75,6,4), 0.513f);
        team.players(players);
        return team;
    }

}
