#Copyright 2012 Joseph Bergin
#License: Creative Commons Attribution-Noncommercial-Share Alike 3.0 United States License

require "robota"
require "robot_exception"


# The the basic features of a world in which the robots move and interact. Other worlds extend this one, 
# such as RobotWorld (in robot_world.rb) and RobotWorld (in tk_robot_world.rb)
class RobotWorldBase
  attr_reader :Beepers, :EastWestWalls, :NorthSouthWalls
  public :Beepers, :EastWestWalls, :NorthSouthWalls
  
  MOVE_ACTION = 0
  TURN_LEFT_ACTION = 1
  PICK_BEEPER_ACTION = 2
  PUT_BEEPER_ACTION = 3
  TURN_OFF_ACTION = 4
  CREATE_ACTION = 5
  NO_ACTION = -1
  
  
 # Create an empty world
 def initialize
   clear
   @Robots = Hash.new(nil)
   @delay = 0.0
 end
 
 # Remove all features (walls, beepers) from the world.  Robots are not affected
 def clear
    @Beepers = Hash.new(0)
    @EastWestWalls = Hash.new(0)
    @NorthSouthWalls = Hash.new(0)
   @Runnables = []
 end
  
  # Assert that the given corner is in the world
  def legal_corner(street, avenue)
    raise IllegalCorner "(#{street}, #{avenue})" if street < 1 || avenue < 1
  end
  
  # Remove all features (walls, beepers, robots) from the world. 
  def reset
    clear
    @Robots = Hash.new(nil)    
  end
  
  # Save the new state of a robot when necessary
  def register_robot(robot, state)
    @Robots[robot] = state
    nil
  end
  
  # Record a robot's actions 
  def update(robot, action, state)
    return if action == NO_ACTION 
    if action == MOVE_ACTION || action == CREATE_ACTION 
      register_robot(robot, state)   
    end
    puts robot.to_s if $tracing
    sleep @delay if @delay > 0
  end
  
  # Set the speed at which the simulation runs 0 = no delay 100 = run very slowly  
  def set_speed (amount)
    amount = 100 - amount
    amount = 0 if amount < 0
    amount = 100 if amount > 100
    @delay = amount / 100.0
  end
  
  # Either (a) pass a number of robots each of which has a run_task method or
  #        (b) pass a block to be executed when the thread starts
  def set_up_thread(*robots, &action)
    if block_given?
      @Runnables << Thread.new{Thread.stop; action.call}
    else
      robots.each {|robot| @Runnables << Thread.new {Thread.stop; robot.run_task}}
    end
  end
  
  # Start all of the known threads and wait until they all complete
  def start_threads(delay = 10)
    delay = 1 if delay <= 0
    sleep(delay / 10.0)
    @Runnables.each{|thread| thread.run}
    @Runnables.each{|thread| thread.join}
  end
  
  # Return true if there are any beepers on the given corner
  def beepers_at?(street, avenue)
    key = [street, avenue]
    return @Beepers[key] != 0
  end
  
  # Return true if there are any robots on the given corner
  def robots_at?(street, avenue)
    @Robots.keys.each do |robot|
      state = @Robots[robot]
      return true if street == state[0] && avenue == state[1]
    end
    return false
  end
  
  # Fail if there are no robots on the given corner
  def assert_robots_at(street, avenue)
    raise NoRobots, "(#{street}, #{avenue})" if ! robots_at?(street, avenue)
  end
  
  # Fail if there are no robots on the given corner
  def assert_beepers_at(street, avenue)
    raise NoBeepers, "(#{street}, #{avenue})" if  ! beepers_at?(street, avenue)
  end
  
  
  # # Read a world from a text file with the given name
  # def read_world(filename)
    # File.open(filename) do |file| file.each_line do |line|
        # if line != "KarelWorld"
          # key, street, avenue, other = line.split
          # street, avenue, other = street.to_i, avenue.to_i, other.to_i
          # if key == "beepers"
            # place_beepers(street, avenue, other)
          # elsif key == "eastwestwalls"
            # while avenue <= other
              # place_wall_north_of(street, avenue)
              # avenue += 1
            # end
          # elsif key == "northsouthwalls"
            # street, avenue = avenue, street
            # while street <= other
              # place_wall_east_of(street, avenue)
              # street += 1
            # end
          # end
        # end
      # end
    # end
    # nil
  # end
#   
  # # Save (write) a world in a text file with a given name
  # def save_world(filename)
    # file = File.new(filename, "w")
    # file.puts "KarelWorld"
    # @Beepers.keys.each{|key| put_line(file, "beepers", key[0], key[1], @Beepers[key])}
    # @EastWestWalls.keys.each{|key| put_line(file, "eastwestwalls", key[0], key[1], key[1])}
    # @NorthSouthWalls.keys.each{|key|  put_line(file, "northsouthwalls", key[1], key[0], key[0])}
    # file.close
    # nil
  # end
#   
  # private  :put_line
#   
  # def put_line(file, key, a, b, c)
    # file.puts "#{key} #{a} #{b} #{c}"
    # nil
  # end
  
  
  # # Place a wall of specified length, North of a given street, crossing a given avenue and proceeding East
  # def place_e_w_wall(northOfStreet, atAvenue, lengthTowardEast)
   # (0..lengthTowardEast-1).each{|i| place_wall_north_of(northOfStreet, atAvenue + i)}
    # nil
  # end
#   
  # # Place a wall of specified length East of a given avenue, crossing a given street and proceeding North
  # def place_n_s_wall(atStreet, eastOfAvenue, lengthTowardNorth)
   # (0..lengthTowardNorth-1).each {|i| place_wall_east_of(atStreet + i, eastOfAvenue)}
    # nil
  # end
  
  # # Return true if there is a wall to the North of the given corner
  # def wall_to_north?(street, avenue)
    # return @EastWestWalls[[street, avenue]] > 0
  # end
#   
  # # Return true if there is a wall to the West of the given corner
  # def wall_to_west?(street, avenue)
    # return (@NorthSouthWalls[[street, avenue - 1]] > 0) || (avenue == 1)
  # end
#   
  # # Return true if there is a wall to the South of the given corner
  # def wall_to_south?(street, avenue)
    # return (@EastWestWalls[[street - 1, avenue]] > 0) || (street == 1)
  # end
#   
  # # Return true if there is a wall to the East of the given corner
  # def wall_to_east?(street, avenue)
    # return @NorthSouthWalls[[street, avenue]] > 0
  # end
# 

  
  # Return the neighbors of the given robot on the given street.
  # Fails if the robot is not on the given corner when invoked
  def neighbors_of(robot, street, avenue)
    result = []
    robot.assert_at(street, avenue)
    @Robots.keys.each do |anyRobot|
      state = @Robots[anyRobot]
      x, y = state[0], state[1]
      result << anyRobot if anyRobot != robot && x == street && y == avenue
    end
    return result
  end
  
    # # make the world visible
  # def set_visible(visible = true)
    # @isVisible = visible
  # end
  # # is the world visible?
  # def visible?
    # return @isVisible
  # end


end