package micro;

/*
 MicroGCL grammar
 <system goal>    -> #Start <program> "EOF_SYMBOL" #Finish
 <program>        -> "BEGIN" <statement list> "END" "."
 <statement list> -> <statement> {<statement>}
 <statement>      -> <variable> ":=" <expression> #Assign ";"
 <statement>      -> "READ"  <var list>  ";"
 <statement>      -> "WRITE"  <expr list> ";" #EndWrite
 <var list>        -> <variable> #ReadVar {"," <variable> #ReadVar }
 <expr list>      -> <expression> #WriteExpr {"," <expression> #WriteExpr}
 <expression>     -> <primary> {<add op> <primary> #AddExpression }
 <primary>        -> "(" <expression> ")"
 <primary>        -> <variable>
 <primary>        -> "INTEGER_LITERAL" 
 <add op>         -> "+"  | "-" 
 <variable>       -> "IDENTIFIER" #ProcessIdentifier
 */

import java.io.BufferedReader;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.PrintWriter;
import java.util.Date;
import java.util.Enumeration;
import java.util.Vector;

// ------------------------------ Token ----------------------------
class Token {
	public static final Token IDENTIFIER 		= new Token("identifier");// 0;
	public static final Token INTEGER_LITERAL 	= new Token("integer literal");// 1;
	public static final Token ASSIGN_OPERATOR 	= new Token(":=");// 2;
	public static final Token PLUS_OPERATOR 	= new Token("+");// 3;
	public static final Token MINUS_OPERATOR 	= new Token("-");// 4;
	public static final Token COMMA 			= new Token(",");// 5;
	public static final Token SEMICOLON 		= new Token(";");// 6;
	public static final Token RIGHT_PARENTHESIS	= new Token(")");// 7;
	public static final Token LEFT_PARENTHESIS 	= new Token("(");// 8;
	public static final Token PERIOD 			= new Token(".");// 9;
	public static final Token BEGIN_SYMBOL 		= new Token("begin");// 10;
	public static final Token END_SYMBOL 		= new Token("end");// 11;
	public static final Token READ_SYMBOL 		= new Token("read");// 12;
	public static final Token WRITE_SYMBOL 		= new Token("write");// 13;
	public static final Token EOF_SYMBOL 		= new Token("eof");// 14;

	public String name() {
		return name;
	}// Only used in error reporting.

	public static String spelling() {
		return Token.buffer;
	}

	static void bufferChar(char c) {
		buffer += c;
	}

	static void clearBuffer() {
		buffer = "";
	}

	private static String buffer = "";
	private String name = null;

	private Token(String name) {
		this.name = name;
	}
}

// ------------------------------ Scanner -------------------------------
class Scanner {
	public Scanner(BufferedReader in, PrintWriter out) {
		this.in = in;
		this.out = out;
	}

	public Token currentToken() {
		if (!tokenAvailable) {
			savedToken = getNextToken();
			tokenAvailable = true;
		}
		return savedToken;
	}

	public void match(Token token) {
		if (token != currentToken()) {
			error(token);
		} else {
			tokenAvailable = false;
		}
	}

	public void toss() {
		tokenAvailable = false;
	} // useful immediately after a check, but DANGEROUS otherwise

	private BufferedReader in;
	private PrintWriter out;
	private String lineBuffer;
	private int lineLength = 0;
	private int linePointer = 0;
	private boolean EOF = false;
	private boolean tokenAvailable = false;
	private Token savedToken = Token.EOF_SYMBOL;
	private int errors = 0;

	private void getNewLine() {
		lineBuffer = "";
		try {
			lineBuffer = in.readLine();
			if (lineBuffer == null) {
				lineBuffer = " ";
				EOF = true;
			}
		} catch (IOException e) {
			EOF = true;
			lineBuffer = " ";
		}
		lineBuffer += ' ';
		lineLength = lineBuffer.length();
		out.println("    %" + lineBuffer);
		linePointer = 0;
	}

	private char inspect() {
		if (linePointer >= lineLength) {
			getNewLine();
		}
		return lineBuffer.charAt(linePointer);
	}

	private void advance() {
		linePointer++;
	}

	private char getNextChar() {
		char c = inspect();
		advance();
		return c;
	}

	private Token checkReserved() {
		if (Token.spelling().equalsIgnoreCase("BEGIN")) {
			return Token.BEGIN_SYMBOL;
		}
		if (Token.spelling().equalsIgnoreCase("END")) {
			return Token.END_SYMBOL;
		}
		if (Token.spelling().equalsIgnoreCase("READ")) {
			return Token.READ_SYMBOL;
		}
		if (Token.spelling().equalsIgnoreCase("WRITE")) {
			return Token.WRITE_SYMBOL;
		}
		return Token.IDENTIFIER;
	}

	private void lexicalError(char c) {
		out.println("Lexical Error Detected at '" + c + "'");
		errors++;
	}

	private void error(Token token) {
		out.println("Match Error Detected Expecting: " + token.name() + " saw: " + savedToken.name());
		errors++;
		System.exit(1);
	}

	static final char blank = ' ';
	static final char tab = '	';

	private Token getNextToken() {
		Token result = Token.EOF_SYMBOL;
		if (EOF) {
			return Token.EOF_SYMBOL;
		}
		Token.clearBuffer();
		boolean finished = false;
		while (!EOF && !finished) {
			char currentChar = getNextChar();
			switch (currentChar) {
			case blank:
			case tab:
				break;
			case 'A': case 'B': case 'C': case 'D': case 'E': case 'F': case 'G':
			case 'H': case 'I': case 'J': case 'K': case 'L': case 'M': case 'N':
			case 'O': case 'P': case 'Q': case 'R': case 'S': case 'T': case 'U':
			case 'V': case 'W': case 'X': case 'Y': case 'Z':
			case 'a': case 'b': case 'c': case 'd': case 'e': case 'f': case 'g':
			case 'h': case 'i': case 'j': case 'k': case 'l': case 'm': case 'n':
			case 'o': case 'p': case 'q': case 'r': case 's': case 't': case 'u':
			case 'v': case 'w': case 'x': case 'y': case 'z': {
				Token.bufferChar(currentChar);
				while (!finished) {
					switch (inspect()) {
					case 'A': case 'B': case 'C': case 'D': case 'E': case 'F': case 'G':
					case 'H': case 'I': case 'J': case 'K': case 'L': case 'M': case 'N':
					case 'O': case 'P': case 'Q': case 'R': case 'S': case 'T': case 'U':
					case 'V': case 'W': case 'X': case 'Y': case 'Z':
					case 'a': case 'b': case 'c': case 'd': case 'e': case 'f': case 'g':
					case 'h': case 'i': case 'j': case 'k': case 'l': case 'm': case 'n':
					case 'o': case 'p': case 'q': case 'r': case 's': case 't': case 'u':
					case 'v': case 'w': case 'x': case 'y': case 'z': 
					case '0': case '1': case '2': case '3': case '4': 
					case '5': case '6': case '7': case '8': case '9':
					case '_': {
						Token.bufferChar(inspect());
						advance();
					}
						break;
					default: {
						result = checkReserved();
						finished = true;
					}
						break;
					}
				}
			}
				break;
			case '0': case '1': case '2': case '3': case '4': 
			case '5': case '6': case '7': case '8': case '9': {
				Token.bufferChar(currentChar);
				while (!finished) {
					switch (inspect()) {
					case '0': case '1': case '2': case '3': case '4': 
					case '5': case '6': case '7': case '8': case '9':{
						Token.bufferChar(inspect());
						advance();
					}
						break;
					default: {
						result = Token.INTEGER_LITERAL;
						finished = true;
					}
						break;
					}
				}
			}
				break;
			case '(': {
				result = Token.LEFT_PARENTHESIS;
				finished = true;
			}
				break;
			case ')': {
				result = Token.RIGHT_PARENTHESIS;
				finished = true;
			}
				break;
			case ';': {
				result = Token.SEMICOLON;
				finished = true;
			}
				break;
			case ',': {
				result = Token.COMMA;
				finished = true;
			}
				break;
			case '+': {
				result = Token.PLUS_OPERATOR;
				finished = true;
			}
				break;
			case ':': {
				if (inspect() == '=') {
					advance();
					result = Token.ASSIGN_OPERATOR;
					finished = true;
				} else {
					lexicalError(inspect());
				}
			}
				break;
			case '.': {
				result = Token.PERIOD;
				finished = true;
			}
				break;
			case '-': {
				if (inspect() == '-') {
					getNewLine();
				} else {
					result = Token.MINUS_OPERATOR;
					finished = true;
				}
			}
				break;
			default: {
				lexicalError(currentChar);
			}
				break;
			}
		}
		if (!finished) {
			result = Token.EOF_SYMBOL;
		}
		return (result);
	}

	public int errors() {
		return errors;
	}
}

// ------------------------------ Parser -------------------------------
class Parser {
	public Parser(PrintWriter out, Scanner scanner, SymbolTable symbolTable,
			CodeGenerator codegen, SemanticActions semantic) {
		this.out = out;
		this.codegen = codegen;
		this.scanner = scanner;
		this.symbolTable = symbolTable;
		this.semantic = semantic;
	}

	public void systemGoal() { // <system goal> -> #Start <program> "EOF_SYMBOL" #Finish
		semantic.start();
		program();
		scanner.match(Token.EOF_SYMBOL);
		semantic.finish();
	}

	private void syntaxError(Token s) {
		out.println("Syntax Error detected. Token was: " + s.name());
		errors++;
	}

	private void program() { // <program> -> "BEGIN" <statement list> "END" "."
		scanner.match(Token.BEGIN_SYMBOL);
		statementList();
		scanner.match(Token.END_SYMBOL);
		scanner.match(Token.PERIOD);
	}

	private void statementList() { // <statement list> -> <statement> {<statement>}
		statement();
		while (scanner.currentToken() == Token.IDENTIFIER
				|| scanner.currentToken() == Token.READ_SYMBOL
				|| scanner.currentToken() == Token.WRITE_SYMBOL) {
			statement();
		}
	}

	private void statement() {
	// <statement> -> <variable> ":=" <expression> #Assign ";"
	// <statement> -> "READ" <var list> ";"
	// <statement> -> "WRITE" <expr list> ";" #EndWrite
		Expression identifier;
		Expression expression;
		if (scanner.currentToken() == Token.IDENTIFIER) {
			identifier = variable();
			scanner.match(Token.ASSIGN_OPERATOR);
			expression = expression();
			semantic.assign(identifier, expression);
			scanner.match(Token.SEMICOLON);
		} else if (scanner.currentToken() == Token.READ_SYMBOL) {
			scanner.toss(); // scanner.match(Token.READ_SYMBOL);
			variableList();
			scanner.match(Token.SEMICOLON);
		} else if (scanner.currentToken() == Token.WRITE_SYMBOL) {
			scanner.toss(); // scanner.match(Token.WRITE_SYMBOL);
			expressionList();
			scanner.match(Token.SEMICOLON);
			semantic.endWrite();
		} else {
			syntaxError(scanner.currentToken());
		}
	}

	private void variableList() { 
		// <var list> -> <variable> #ReadVar {"," <variable> #ReadVar }
		Expression variable = variable();
		semantic.readVariable(variable);
		while (scanner.currentToken() == Token.COMMA) {
			scanner.toss();
			variable = variable();
			semantic.readVariable(variable);
		}
	}

	private void expressionList() { 
		// <expr list> -> <expression> #WriteExpr {"," <expression> #WriteExpr}
		Expression expression = expression();
		semantic.writeExpression(expression);
		while (scanner.currentToken() == Token.COMMA) {
			scanner.toss();
			expression = expression();
			semantic.writeExpression(expression);
		}
	}

	private Expression expression() { 
		// <expression> -> <primary> {<add op> <primary> #AddExpression}
		Expression rightOperand;
		AddOperator operator;
		Expression result = primary();
		while (scanner.currentToken() == Token.PLUS_OPERATOR
				|| scanner.currentToken() == Token.MINUS_OPERATOR) {
			operator = addOperator();
			rightOperand = primary();
			result = semantic.addExpression(result, operator, rightOperand);
		}
		return result;
	}

	private Expression primary() {
	// <primary> -> "(" <expression> ")" | <variable | "INTEGER_LITERAL"
		Expression result = null;
		if (scanner.currentToken() == Token.LEFT_PARENTHESIS) {
			scanner.toss();
			result = expression();
			scanner.match(Token.RIGHT_PARENTHESIS);
		} else if (scanner.currentToken() == Token.IDENTIFIER) {
			result = variable();
		} else if (scanner.currentToken() == Token.INTEGER_LITERAL) {
			result = new LiteralExpression(Token.spelling());
			scanner.match(Token.INTEGER_LITERAL);
		} else {
			syntaxError(scanner.currentToken());
		}
		return result;
	}

	private AddOperator addOperator() { // <add op> -> "+" | "-"
		Token token = scanner.currentToken();
		AddOperator result = null;
		if (token == Token.PLUS_OPERATOR) {
			result = AddOperator.PLUS;
			scanner.match(token);
		} else if (token == Token.MINUS_OPERATOR) {
			result = AddOperator.MINUS;
			scanner.match(token);
		} else {
			syntaxError(token);
		}
		return result;
	}

	private Expression variable() { // <variable> -> "IDENTIFIER" #ProcessIdentifier
		scanner.currentToken(); // update the spelling
		Expression result = semantic.processIdentifier(new IdExpression(Token
				.spelling()));
		scanner.match(Token.IDENTIFIER);
		return result;
	}

	public int errors() {
		return errors;
	}

	private PrintWriter out;
	private CodeGenerator codegen;
	private Scanner scanner;
	private SymbolTable symbolTable;
	private SemanticActions semantic;
	private int errors = 0;
}

// ------------------------------ SemanticActions ------------------------
// ------------------------- First the Semantic Records --------------------
// --- These are the params and return values of parser and semantic methods
// ------------------------------ AddOperator ----------------------------
class AddOperator { // Typed enumeration.
	private AddOperator(String samCode) {
		code = samCode;
	}

	public static final AddOperator PLUS = new AddOperator("IA      ");
	public static final AddOperator MINUS = new AddOperator("IS      ");

	public String samCode() {
		return code;
	}

	private String code;
}

// ------------------------------ Expression --------------------------
interface Expression {
	public abstract String samCode();
}

class IdExpression implements Expression { // Represents an identifier: Immutable
	public IdExpression(String name) {
		this.name = name;
	}

	public String samCode() {
		return "$" + name + "$";
	}

	private String name; // The spelling of the identifier
}

class LiteralExpression implements Expression { // Represents a numeric literal: Immutable
	public LiteralExpression(String value) {
		this.value = Integer.parseInt(value);
	}

	public String samCode() {
		return "#" + value;
	};

	private int value;// The literal value.
}

class TemporaryExpression implements Expression { // Represents a cpu register: Immutable
	public TemporaryExpression(int which) {
		this.which = which;
	}

	public String samCode() {
		return "R" + which;
	}

	public int which() {
		return which;
	}

	private int which; // The register number
}

// ------------------------------ SemanticActions Class ------------------
class SemanticActions {
	public SemanticActions(PrintWriter out, CodeGenerator codegen,
			SymbolTable idtable) {
		this.out = out;
		this.codegen = codegen;
		this.idtable = idtable;
	}

	void semanticError(String s) {
		out.println("SemanticActions Error: " + s);
		errors++;
		System.exit(1);
	}

	public void start() {
		out.println("    %  Compiled on " + new Date());
		out.println("    %  Author(s): Joseph Bergin");
	}

	public void finish() {
		codegen.generate0Address("HALT");
		codegen.generateVariables();
	}

	public Expression processIdentifier(IdExpression spelling) {
		idtable.checkIdentifier(spelling.samCode());
		return spelling;
	}

	public Expression addExpression(Expression left, AddOperator addOp,
			Expression right) {
		TemporaryExpression register = codegen.loadRegister(left);
		codegen.generate2Address(addOp.samCode(), register, right);
		codegen.freeTemporary(right);
		return register;
	}

	public void assign(Expression target, Expression source) {
		TemporaryExpression register = codegen.loadRegister(source);
		codegen.generate2Address("STO     ", register, target);
		codegen.freeTemporary(register);
		codegen.freeTemporary(source);
		codegen.freeTemporary(target);
	}

	public void writeExpression(Expression outExpression) {
		codegen.generate1Address("WRI     ", outExpression);
		codegen.freeTemporary(outExpression);
	}

	public void endWrite() {
		codegen.generate0Address("WRNL");
	}

	public void readVariable(Expression inVariable) {
		codegen.generate1Address("RDI     ", inVariable);
	}

	public int errors() {
		return errors;
	}

	private PrintWriter out;
	private CodeGenerator codegen;
	private SymbolTable idtable;
	private int errors = 0;
}

// ------- CodeGenerator -- See the SAM documentation ---------------------
class CodeGenerator {
	public CodeGenerator(PrintWriter out, SymbolTable idtable) {
		this.out = out;
		this.idtable = idtable;
		for (int i = 0; i < totalRegisters; ++i) {
			freeRegisters[i] = true;
		}
	}

	public TemporaryExpression getTemporary() { // There are exactly 16 registers.
		int register = 0;
		while (register < totalRegisters && !freeRegisters[register]) {
			register++;
		}
		if (register < totalRegisters) {
			freeRegisters[register] = false;
		}
		return new TemporaryExpression(register);
	}

	public TemporaryExpression loadRegister(Expression expression) {
		if (expression instanceof TemporaryExpression) {
			return (TemporaryExpression) expression;
		}
		TemporaryExpression register = getTemporary();
		generate2Address("LD      ", register, expression);
		return register;
	}

	public void registerReport() {
		out.print("  --    Allocated registers:");
		System.out.print("  --    Allocated registers:");
		String fin = " None";
		for (int i = 0; i < totalRegisters; ++i) {
			if (!freeRegisters[i]) {
				out.print(" " + i);
				System.out.print(" " + i);
				fin = " <- Find and fix.";
			}
		}
		out.println(fin);
		System.out.println(fin);
	}

	public void freeTemporary(Expression expression) {
		if (expression instanceof TemporaryExpression) {
			freeRegisters[((TemporaryExpression) expression).which()] = true;
		}
	}

	public void generate0Address(String opcode) {
		out.println("    " + opcode);
	}// Generate a 0 address SAM instruction like halt

	public void generate1Address(String opcode, Expression arg) {
		out.println("    " + opcode + arg.samCode());
	} // generate 1 address SAM instruction like rdi

	public void generate2Address(String opcode, TemporaryExpression arg1, Expression arg2) {
		out.println("    " + opcode + arg1.samCode() + ",  " + arg2.samCode());
	}// generate a 2 addr SAM instruction like ia

	public void generateVariables() { // generate code for the variable block at the end.
		Enumeration<String> variables = idtable.elements();
		while (variables.hasMoreElements()) {
			out.println("LABEL    " + variables.nextElement());
			out.println("     SKIP     2");
		}
	}

	private PrintWriter out;
	private static final int totalRegisters = 16;
	private boolean freeRegisters[] = new boolean[totalRegisters];
	private SymbolTable idtable;
}

// ------------------------------ SymbolTable -------------------------------

class SymbolTable { // Too simple to be realistic, actually.
	public SymbolTable(PrintWriter messages) {
		out = messages;
	}

	public int lookUp(String s) {
		for (int i = 0; i < symbolTable.size(); ++i) {
			if (s.equals(symbolTable.elementAt(i))) {
				return i;
			}
		}
		return -1;
	}

	private int enter(String s) {
		symbolTable.addElement(s);
		int result = symbolTable.size() - 1;
		return result;
	}

	public void checkIdentifier(String s) {
		int where = lookUp(s);
		if (where < 0) {
			where = enter(s);
			out.println("    %    Implicit declaration of: "
					+ s.substring(1, s.length() - 1));
		}
	}

	public Enumeration<String> elements() {
		return symbolTable.elements();
	}

	private Vector<String> symbolTable = new Vector<String>();
	private PrintWriter out;
}

// --------------------------- MicroGCLCompiler ---------------------------
public class MicroGCLCompiler {
	public static void main(String[] args) {
		if (args.length < 2) {
			BufferedReader inp = new BufferedReader(new InputStreamReader( System.in));
			String[] temp = new String[2];
			if (args.length < 1) {
				System.out.println("Enter the input fliename");
				try {
					temp[0] = inp.readLine();
				} catch (IOException ex) {
					System.out.println("Error reading input filename.");
				}
			} else {
				temp[0] = args[0];
			}
			System.out.println("Enter the listing filename [codefile]");
			try {
				temp[1] = "codefile";
				String in = inp.readLine();
				if (in != null && !in.equals("")) {
					temp[1] = in;
				}
			} catch (IOException ex) {
				System.out.println("Error reading listing filename.");
			}
			args = temp;
		}
		try {
			BufferedReader sourceFile = new BufferedReader(new FileReader( args[0]));
			PrintWriter listingFile = new PrintWriter(new FileWriter(args[1]), true);

			Scanner scanner = new Scanner(sourceFile, listingFile);
			SymbolTable idtable = new SymbolTable(listingFile);
			CodeGenerator codegen = new CodeGenerator(listingFile, idtable);
			SemanticActions semantic = new SemanticActions(listingFile, codegen, idtable);
			Parser parser = new Parser(listingFile, scanner, idtable, codegen, semantic);
			parser.systemGoal();
			listingFile.print("     %    End of Compilation");
			codegen.registerReport();
			int totalErrors = scanner.errors() + parser.errors() + semantic.errors();
			String errorMessage;
			switch (totalErrors) {
			default: {
				errorMessage = "were " + totalErrors + " errors.";
			}
				break;
			case 0: {
				errorMessage = "were no errors.";
			}
				break;
			case 1: {
				errorMessage = "was 1 error.";
			}
				break;
			}
			System.out.println("Done. There " + errorMessage);
			listingFile.println("     %    There " + errorMessage);
		} catch (IOException e) {
			System.out.println("File errors: " + e);
			System.exit(1);
		}
	}
}
