package orderprocessing;

/*
 History
 date		description
 -----		-----------
 11/12/99	Creation of 2 billing terms.
 8/1/00		More terms added + infrastructure
 */

/** This class represents various billing terms in use by the firm
 * it exports constant terms objects for all allowable terms.
 */
public class BillingTerms implements java.io.Serializable {
	
	private static final int NET = 0;

	private static final int EOM = 1;

	private static final BillingTerms NET_10_DAYS = new BillingTerms(NET, 10);

	private static final BillingTerms NET_30_DAYS = new BillingTerms(NET, 30);

	private static final BillingTerms EOM_10_DAYS = new BillingTerms(EOM, 10);

	private int terms = NET;

	private int days = 0;

	/** retrieve the terms code for these billing terms
	 * NET = 0
	 * EOM = 1
	 * @return the terms code
	 */
	public int terms() {
		return terms;
	}

	/** Retrieve the number of days in these terms, usually 10 or 30
	 * @return the number of days
	 */
	public int days() {
		return days;
	}

	/** Retrieve a string representation of these terms
	 * @return the terms as a string
	 */
	public String code() {
		String pre = terms == NET ? "NET " : "EOM ";
		return pre + days;
	}

	/** Retrieve a terms object from its string representation
	 * "net 30", "net 10" and "eom 10" are recognized. NOT case sensitive.
	 * An illegal entry will result in "net 30" by default
	 * @param code the string representation of a billing terms
	 * @return the terms object
	 */
	//If there were many more of these, a HashMap might be preferable to this search.
	public static BillingTerms get(String code) {
		if (code.toUpperCase().equals("NET 10")) {
			return NET_10_DAYS;
		}
		if (code.toUpperCase().equals("EOM 10")) {
			return EOM_10_DAYS;
		}
		return NET_30_DAYS; // default value
	}

	/** return the due date for payment on the associated order given the invoice
	 * date.
	 * @param invoiceDate the date that the order was invoiced.
	 * @return the due date for payment according to these terms.
	 */
	public CalculationCalendar dueDate(CalculationCalendar invoiceDate) {
		CalculationCalendar result = (CalculationCalendar) invoiceDate.clone();
		if (terms == NET) {
			result.add(CalculationCalendar.DATE, days);
		} else { // TODO ask the Client
		}
		return result;
	}

	private BillingTerms(int terms, int days) {
		this.terms = terms;
		this.days = days;
	}

	public String toString() {
		return "Billing terms " + code();
	}

	public boolean equals(Object other) {
		if (!(other instanceof BillingTerms)) {
			return false;
		}
		BillingTerms o = (BillingTerms) other;
		return terms == o.terms && days == o.days;
	}

	public int hashCode() {
		return terms * 997 + days * 13;
	}

	/** Display this billing terms info on standard output
	 * 
	 */
	public void display() {
		System.out.println(this);
	}


}
