package orderprocessing;

import java.util.GregorianCalendar;
import java.util.Locale;
import java.util.TimeZone;

/** This class provides an increment function for Gregorian Calendars so that a
 *	future date may be easily calculated. It can also tell you which of two dates
 *	is earlier by returning the earlier (or later) one to you.
 */

public class CalculationCalendar extends GregorianCalendar implements
		java.io.Serializable {
	public CalculationCalendar() {
		super();
	}

	/**
	 * Constructs a CalculationCalendar based on the current time
	 * in the given time zone with the default locale.
	 * @param zone the given time zone.
	 */
	public CalculationCalendar(TimeZone zone) {
		super(zone);
	}

	/**
	 * Constructs a CalculationCalendar based on the current time
	 * in the default time zone with the given locale.
	 * @param aLocale the given locale.
	 */
	public CalculationCalendar(Locale aLocale) {
		super(aLocale);
	}

	/**
	 * Constructs a CalculationCalendar based on the current time
	 * in the given time zone with the given locale.
	 * @param zone the given time zone.
	 * @param aLocale the given locale.
	 */
	public CalculationCalendar(TimeZone zone, Locale aLocale) {
		super(zone, aLocale);
	}

	/**
	 * Constructs a CalculationCalendar with the given date set
	 * in the default time zone with the default locale.
	 * @param year the value used to set the YEAR time field in the calendar.
	 * @param month the value used to set the MONTH time field in the calendar.
	 * Month value is 0-based. e.g., 0 for January.
	 * @param date the value used to set the DATE time field in the calendar.
	 */
	public CalculationCalendar(int year, int month, int date) {
		super(year, month, date);
	}

	/**
	 * Constructs a CalculationCalendar with the given date
	 * and time set for the default time zone with the default locale.
	 * @param year the value used to set the YEAR time field in the calendar.
	 * @param month the value used to set the MONTH time field in the calendar.
	 * Month value is 0-based. e.g., 0 for January.
	 * @param date the value used to set the DATE time field in the calendar.
	 * @param hour the value used to set the HOUR_OF_DAY time field
	 * in the calendar.
	 * @param minute the value used to set the MINUTE time field
	 * in the calendar.
	 */
	public CalculationCalendar(int year, int month, int date, int hour,
			int minute) {
		super(year, month, date, hour, minute);
	}

	/**
	 * Constructs a CalculationCalendar with the given date
	 * and time set for the default time zone with the default locale.
	 * @param year the value used to set the YEAR time field in the calendar.
	 * @param month the value used to set the MONTH time field in the calendar.
	 * Month value is 0-based. e.g., 0 for January.
	 * @param date the value used to set the DATE time field in the calendar.
	 * @param hour the value used to set the HOUR_OF_DAY time field
	 * in the calendar.
	 * @param minute the value used to set the MINUTE time field
	 * in the calendar.
	 * @param second the value used to set the SECOND time field
	 * in the calendar.
	 */
	public CalculationCalendar(int year, int month, int date, int hour,
			int minute, int second) {
		super(year, month, date, hour, minute, second);
	}

	/** Increments a calendar by an amount dependent on the chosen date field
	 * @param field which field should be incremented
	 * @param amount the amount to increment "field" by.
	 * @return a new calendar suitably incremented.
	 */
	public CalculationCalendar incrementedBy(int field, int amount) {
		CalculationCalendar result = (CalculationCalendar) this.clone();
		result.add(field, amount);
		return result;
	}

	/** Returns the maximum (later) of two dates. If the parameter is null
	 * it returns this.
	 * @param other a calendar to compare with this.
	 * @return the later of other and this.
	 */
	public CalculationCalendar max(CalculationCalendar other) {
		if (other == null) {
			return this;
		}
		CalculationCalendar result = this;
		if (result.before(other)) {
			result = other;
		}
		return result;
	}

	/** Returns the minimum (earlier) of two dates. If the parameter is null
	 * it returns this.
	 * @param other a calendar to compare with this.
	 * @return the earlier of other and this.
	 */
	public CalculationCalendar min(CalculationCalendar other) {
		if (other == null) {
			return this;
		}
		CalculationCalendar result = this;
		if (other.before(this)) {
			result = other;
		}
		return result;
	}

}