package orderprocessing;

import java.io.Serializable;
import java.util.HashMap;
import java.util.Iterator;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;

/*
 History
 date		description
 -----		-----------
 08/27/07	Modifed from the original, create separate address class

 */

/** This class maintains information about a single customer
 <p>
 Note that any changes to the public part of this class or any change in
 the fields will require versioning. This is because these objects are
 stored in the object database for this application.
 <p>
 This class can serve as a model for other classes that need versioning. Be
 sure to add versioning infrastructure to them before you include them in
 the database for the first time.
 <p>
 Note that Addresses are already implicitly included in the database since they
 are written whenever a Customer is written.
 <p>
 Additionally, this class maintains a collection of all customers.
 */
public class Customer implements Serializable {
	
	private String key;

	private String name;

	private Address billingAddress;

	private Address shipAddress;

	private BillingTerms terms = BillingTerms.get("NET 30");

	private boolean isActive = true;

	/** create a customer object with default values for ship address and terms
	 */
	public Customer(String name, Address billingAddress) {
		this(name, billingAddress, billingAddress, BillingTerms.get("NET 30"));
	}

	/** create a customer object with default values for ship address 
	 */
	public Customer(String name, Address billingAddress,
			BillingTerms billingTerms) {
		this(name, billingAddress, billingAddress, billingTerms);
	}

	/** Create a customer object with default terms.
	 */
	public Customer(String name, Address billingAddress, Address shipAddress) { // To Do: error checking for id and name
		this(name, billingAddress, shipAddress, BillingTerms.get("Net 30"));
	}

	/** create a customer object with all options specified.
	 * @param name the customer name.
	 * @param billingAddress the customer billing address
	 * @param shipAddress the ship address if not specified on an order
	 * @param billingTerms the billing terms if not specified on an order.
	 * 
	 * Note: As with ALL constructors in ALL classes, the verification of the data
	 * must occur BEFORE you call the constructor. Only call a constructor when
	 * the data is correct to create a fully usable object that maintains all
	 * of its invariants. Never create a partially complete object unless you take
	 * foolproof steps to complete the process of creation. 
	 */
	public Customer(String name, Address billingAddress, Address shipAddress,
			BillingTerms billingTerms) {
		this.name = name;
		this.billingAddress = billingAddress;
		this.shipAddress = shipAddress;
		this.terms = billingTerms;
		key = IDGenerator.nextID();

		allCustomers.put(key, this);
	}

	// Accessors

	/** Retrieve the customer number
	 * @return the customer number
	 */
	public String key() {
		return key;
	}

	/**Retrieve the customer name
	 * @return the customer's name
	 */
	public String name() {
		return name;
	}

	/** Retrieve the customer's billing address
	 * @return the customer's billing address
	 */
	public Address billingAddress() {
		return billingAddress;
	}

	/** Retrieve the customer's ship address
	 * @return the customer's shipping address
	 */
	public Address shippingAddress() {
		return shipAddress;
	}

	/** Retrieve this customer's default billing terms
	 * @return the billing terms
	 */
	public BillingTerms billingTerms() {
		return terms;
	}

	// Mutators

	/** Set new billing terms for this customer
	 * @param terms the new billing terms
	 */
	public void billingTerms(BillingTerms terms) {
		this.terms = terms;
	}

	/**  Change the name of this customer
	 * @param name the new customer name
	 */
	public void name(String name) {
		this.name = name;
		billingAddress.name(name); //avoid redundancy.
	}

	/* Note that you cannot change the (reference to) address objects in a 
	 * customer object, but addresses are mutable. 
	 */

	// other methods
	/** Display this customer info on standard output
	 * 
	 */
	public void display() {
		System.out.println(this);
	}

	public boolean equals(Object other) {
		if (!(other instanceof Customer))
			return false;
		return key.equals(((Customer) other).key);
	}

	public int hashcode() {
		return key.hashCode() * 997;
	}

	public String toString() {
		return "\t\t\t" + " Customer ID :" + key + "\n" + " Name    : " + name
				+ "\n" + " Address : " + billingAddress + "\n"
				+ " Ship Address : \n" + shipAddress + "\n";
	}


	private static class IDGenerator {
		public static String nextID() {
			return ID_PREFIX + Long.toString(nextID++);
		}

		private static long nextID = 5001;

		private final static String ID_PREFIX = "C";
	}

	private static HashMap<String, Customer> allCustomers = new HashMap<String, Customer>();

	/** Write all of the customer data for all customers to an output stream
	 * @param out the Object stream to write to
	 * @throws IOException
	 */
	public static final void close(ObjectOutputStream out) throws IOException {
		out.writeLong(IDGenerator.nextID);
		out.writeObject(allCustomers);
	}

	/**  Tell if a customer is active. A customer is active until "removed."
	 * @return true if the customer is active. 
	 */
	public boolean isActive() {
		return isActive;
	}

	/** Read all customer data for all customers into the program at
	 * the beginning of a run
	 * @param in the Object stream to read from
	 * @throws IOException
	 * @throws ClassNotFoundException
	 */
	public static final void open(ObjectInputStream in) throws IOException,
			ClassNotFoundException {
		IDGenerator.nextID = in.readLong();
		allCustomers = (HashMap<String, Customer>) in.readObject();
		//System.out.println(IDGenerator.nextID);
	}

	/** Provide an iteration service over all known customers. The iterator
	 * will return customer objects. There is no implied ordering. 
	 * @return an Iterator over the customer part of the database
	 */
	public static Iterator<Customer> customers() {
		return allCustomers.values().iterator();
	}

	/** Provide an iterator over the customer numbers.  The iterator will
	 * return just the customer numbers as Strings. There is no implied ordering. 
	 * @return an iterator over the customer numbers (strings). 
	 */
	public static Iterator<String> keys() {
		return allCustomers.keySet().iterator();
	}

	/** Display all of the customers on standard output. No order implied. 
	 * 
	 */
	public static void dump() {
		System.out.println("\n\n\n Customers \n\n\n");
		System.out.println(allCustomers);
	}

	/** Get a particular customer object from its customer number. 
	 * @param customerNumber the customer's number
	 * @return the customer object or null if none
	 */
	public static Customer get(String customerNumber) {
		Customer customer = allCustomers.get(customerNumber);
		if (customer == null || customer.isActive) {
			return customer;
		}
		return null;
	}

	/** Make a customer inactive. 
	 * @param key the customer to "remove"
	 */
	public static void remove(String key) {
		Customer c = allCustomers.get(key);
		if (c != null) {
			c.isActive = false;
		}
	}

}