package orderprocessing;

/*
 History
 date		description
 -----		-----------
 11/12/99	Change date to CalculationCalendar,
 format output
 */

import java.util.ArrayList;
import java.util.Iterator;
import java.io.Serializable;
import java.text.NumberFormat;
import java.util.HashMap;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.BufferedWriter;

/** This class represents what will ship from a customer order.
 *	<p>
 *	Additionally, this class maintains a collection of all Invoices.
 */

public class Invoice implements Serializable {
	
	private ArrayList<OrderItem> shipableItems = null; // ArrayList of OrderItems

	private String key;

	private CalculationCalendar invoiceDate = null;

	private CalculationCalendar shipDate;

	private Order order;


	/** Create an Invoice for a particular customer and a sequence of items ordered
	 * @param order The order from which this invoice originates
	 * @param shipableItems A ArrayList of OrderItems representing the items shipped
	 */
	public Invoice(Order order, ArrayList<OrderItem> shipableItems) {
		this.shipableItems = (ArrayList<OrderItem>) shipableItems.clone();
		shipDate = order.requestedShippingDate();
		this.order = order;
		key = IDGenerator.nextID();
		allInvoices.put(key, this);
		invoiceDate = shipDate();
	}

	// accessors

	/** Retrieve the customer that generated this invoice
	 * @return the customer object
	 */
	public Customer customer() {
		return order.customer();
	}

	/** Iterate over the shippable items for this invoice
	 * @return an iterator over shippable items (no implied order)
	 */
	public Iterator<OrderItem> shipableItems() {
		return shipableItems.iterator();
	}

	/** Retrieve the number of line items on this invoice
	 * @return the number of line items. 
	 */
	public int size() {
		return shipableItems.size();
	}

	/**  Retrieve the date this invoice will/did ship
	 * @return the invoice ship date
	 */
	public CalculationCalendar shipDate() {
		return shipDate;
	}

	/** Retrieve the date the invoice was generated
	 * @return the invoice generation date
	 */
	public CalculationCalendar invoiceDate() {
		return invoiceDate;
	}

	/** Retrieve the order that generated this invoice
	 * @return the order 
	 */
	public Order order() {
		return order;
	}

	/** Return the invoice number
	 *	@return the invoice number
	 */
	public String key() {
		return key;
	}

	/** Retrieve the date payment is due for this invoice
	 * @return the payment due date
	 */
	public CalculationCalendar paymentDueDate() {
		return order.billingTerms().dueDate(invoiceDate); // is this OK?
	}

	/** Display on standard out the information about Items that 
	 * will ship, including extended prices and a total
	 */
	public void display() {
		System.out.println("\t\t\t------- Invoice -------");
		System.out.println("Invoice No: " + key);
		//System.out.println("Customer : " );
		order.customer().display();
		System.out.println("Invoice Date: " + invoiceDate.getTime());
		Iterator<OrderItem> items = shipableItems();
		double total = 0.0;

		System.out.println("\n\n\tQty\tItem No. "
				+ "\tDescription\t\tsub-total\n");

		while (items.hasNext()) {
			OrderItem ordered = items.next();
			ordered.displayExtended();
			total += ordered.quantity() * ordered.unitPrice();
		}

		System.out.println("\n\t\t\t\tInvoice Total: "
				+ NumberFormat.getInstance().format(total));
		System.out
				.println("-----------------------------------------------------------------");
	}

	public boolean equals(Object other) {
		if (!(other instanceof Invoice))
			return false;
		return key.equals(((Invoice) other).key);
	}

	public int hashCode() {
		return key.hashCode() * 997;
	}

	public String toString() {
		return "Invoice with key: " + key;
	}

	/** Format all of the invoices (no implied order) and write to output
	 * @param file the file to write to
	 */
	public static void formatAll(BufferedWriter file) // NOTE: NOT in invoice number order.
	{
		Iterator<Invoice> invoices = invoices();
		Invoice invoice = null;
		while (invoices.hasNext()) {
			try {
				invoice = invoices.next();
				invoice.format(file);
				file.newLine();
				file.newLine();
			} catch (IOException ex) {
				System.out.println("Failed to write invoice " + invoice.key()
						+ "  " + ex);
			}
		}
	}

	// To do this in invoice number order you can iterate over the keys to extract them
	// into an ArrayList. Sort that list and then use the sorted list to get the keys
	// in sorted order to get the records themselves.

	/** Format the invoices for printing
	 * @param file the file to write the formatted invoice on
	 * @throws IOException
	 */

	public void format(BufferedWriter file) throws IOException { // TODO format invoices
	}

	private static class IDGenerator {
		public static String nextID() {
			return ID_PREFIX + Long.toString(nextID++);
		}

		private static long nextID = 6001;

		private final static String ID_PREFIX = "I";
	}

	// Following is the collection of all invoices.
	private static HashMap<String, Invoice> allInvoices = new HashMap<String, Invoice>();

	/** Get a particular invoice object from its customer number. 
	 * @param key the invoice number
	 * @return the invoice object or null if none
	 */
	public static Invoice get(String key) {
		return allInvoices.get(key);
	}

	/** Write all of the Invoice data for all invoices to an output stream
	 * @param out the Object stream to write to
	 * @throws IOException
	 */
	public static final void close(ObjectOutputStream out) throws IOException {
		out.writeLong(IDGenerator.nextID);
		out.writeObject(allInvoices);
	}

	/** Read all invoice data for all invoices into the program at
	 * the beginning of a run
	 * @param in the Object stream to read from
	 * @throws IOException
	 * @throws ClassNotFoundException
	 */
	public static final void open(ObjectInputStream in) throws IOException,
			ClassNotFoundException {
		IDGenerator.nextID = in.readLong();
		allInvoices = (HashMap<String, Invoice>) in.readObject();
	}

	/** Provide an iteration service over all known invoices. The iterator
	 * will return invoice objects. There is no implied ordering. 
	 * @return an Iterator over the invoice part of the database
	 */
	public static Iterator<Invoice> invoices() {
		return allInvoices.values().iterator();
	}

	/** Provide an iterator over the invoice numbers.  The iterator will
	 * return just the invoice numbers as Strings. There is no implied ordering. 
	 * @return an iterator over the invoice numbers (strings). 
	 */
	public static Iterator<String> keys() {
		return allInvoices.keySet().iterator();
	}

	/** Display all of the invoices on standard output. No order implied. 
	 * 
	 */
	public static void dump() {
		System.out.println("\n\n\n Invoices \n\n\n");
		System.out.println(allInvoices);
	}
}