package orderprocessing;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.Iterator;

class Maintenance {
	/**
	 * Writes all customer records to a file in a format from which they can be
	 * restored with the restore function. They are written in customer number
	 * order.
	 * 
	 * @param out
	 *            the file to write to
	 */
	private static void dumpCustomers(BufferedWriter out) {
		try {
			ArrayList<String> list = sortedKeys(Customer.keys());
			Iterator<String> it = list.iterator();
			Customer c;
			while (it.hasNext()) {
				c = Customer.get(it.next());
				out.write(c.key());
				out.newLine();
				out.write(c.name());
				out.newLine();
				out.write(("" + c.billingAddress()).replace('\n', '~'));
				out.newLine();
				out.write(("" + c.shippingAddress()).replace('\n', '~'));
				out.newLine();
				out.write("" + c.billingTerms());
				out.newLine();
				out.write("" + c.isActive());
				out.newLine();
				out.newLine();// separator
			}
		} catch (IOException ex) {
			System.out.println("Customer Write Failed. " + ex);
		}
	}

	/**
	 * Returns a sorted list of the results (strings) returned by the iterator.
	 * Requires that the iterator return Comparable items.
	 * 
	 * @param i
	 *            the source of records to be sorted
	 * @return a sorted list of keys.
	 */
	public static ArrayList<String> sortedKeys(Iterator<String> i) {
		ArrayList<String> result = new ArrayList<String>();
		while (i.hasNext()) {
			result.add(i.next());
		}
		Collections.sort(result);
		return result;
	}

	/**
	 * Returns a sorted list of the results returned by the iterator. Sorted in
	 * order of the Comparator.
	 * 
	 * @param i
	 *            the iterator source of the keys
	 * @param c
	 *            the comparison object used to sort
	 * @return a sorted list of the keys
	 */
	public static ArrayList<String> sortedKeys(Iterator<String> i, Comparator<String> c) {
		ArrayList<String> result = new ArrayList<String>();
		while (i.hasNext()) {
			result.add(i.next());
		}
		Collections.sort(result, c);
		return result;
	}

	/**
	 * Adds customers from a file. Note that the customer number line is
	 * required but not used. The Customer class assigns customer numbers. init
	 * the database first if you want the results of the file to replace the
	 * objects in the database. In this case the records should be in the order
	 * required.
	 * 
	 * @param in
	 *            the file to read from
	 */
	private static void restoreCustomers(BufferedReader in) {
		try {
			String num;
			while ((num = in.readLine()) != null && !num.equals("PRODUCTS")) {
				String name = in.readLine();
				String cAddress = in.readLine();
				String sAddress = in.readLine();
				String bt = in.readLine().substring(14);
				Address custAddress = new Address(cAddress, "~");
				Address shipAddress = new Address(sAddress, "~");
				BillingTerms terms = BillingTerms.get(bt);
				Customer c = new Customer(name, custAddress, shipAddress, terms);
				boolean active = in.readLine().equals("true") ? true : false;
				if (!active) {
					Customer.remove(c.key());
				}

				in.readLine(); // separator line
			}
		} catch (IOException ex) {
			System.out.println("Customer Restore Failed. " + ex);
			System.exit(1);
		}
	}

	/**
	 * creates an empty database. All records are lost.
	 */
	private static void initDatabase() {
		OrderProcessor.close(OrderProcessor.databaseName());
	}

	/**
	 * Write all orders to a readable stream
	 * 
	 * @param out
	 *            the file to write to
	 */
	private static void dumpOrders(BufferedWriter out) {
		try {
			ArrayList<String> list = sortedKeys(Order.keys());
			Iterator<String> it = list.iterator();
			Order order;
			out.write("ORDERS");
			out.newLine();
			while (it.hasNext()) {
				order = Order.get(it.next());
				out.write(order.key());
				out.newLine();
				out.write(order.customer().key());
				out.newLine();
				out.write(order.billingTerms().toString().substring(14));
				out.newLine();
				out.write(order.billingAddress().toString().replace('\n', '~'));
				out.newLine();
				out
						.write(order.shippingAddress().toString().replace('\n',
								'~'));
				out.newLine();
				out.write(sdf.format(order.orderDate().getTime()));
				out.newLine();
				out.write(sdf.format(order.requestedShippingDate().getTime()));
				out.newLine();
				out.write(sdf.format(order.cancelDate().getTime()));
				out.newLine();
				out.write("" + order.size());
				out.newLine();
				Iterator<OrderItem> j = order.orderedItems();
				while (j.hasNext()) {
					OrderItem item = j.next();
					dumpOrderItem(item, out);
				}
				out.newLine();
			}
		} catch (IOException ex) {
			System.out.println("Order Write Failed. " + ex);
		}
	}

	/**
	 * Dump a single order item to the readable file
	 * 
	 * @param o
	 *            the order item to write
	 * @param out
	 *            the stram to write on
	 * @throws IOException
	 */
	private static void dumpOrderItem(OrderItem o, BufferedWriter out)
			throws IOException {
		out.write("" + o.itemNumber() + "~" + o.quantity() + "~"
				+ o.unitPrice() + "~" + o.description());
		out.newLine();
	}

	/**
	 * Restore an order item from the readable file so it can be put into the in
	 * memory database
	 * 
	 * @param in
	 *            the file to read from
	 * @return the order item read
	 * @throws IOException
	 */
	private static OrderItem restoreOrderItem(BufferedReader in)
			throws IOException {
		String line = in.readLine();
		return OrderItem.generateOrderItem(line, "~");
	}

	/**
	 * Restore all orders from the readable (backup) file
	 * 
	 * @param in
	 *            the stream to read from
	 */
	private static void restoreOrders(BufferedReader in) {
		try {
			String num;
			while ((num = in.readLine()) != null && !num.equals("INVOICES")) {
				Customer cust = Customer.get(in.readLine().trim());

				BillingTerms terms = BillingTerms.get(in.readLine().trim());
				Address billingAddress = new Address(in.readLine().trim(), "~");
				Address shipAddress = new Address(in.readLine().trim(), "~");
				CalculationCalendar od = new CalculationCalendar();
				CalculationCalendar sd = new CalculationCalendar();
				CalculationCalendar cd = new CalculationCalendar();

				try {
					od.setTime(sdf.parse(in.readLine().trim()));
					sd.setTime(sdf.parse(in.readLine().trim()));
					cd.setTime(sdf.parse(in.readLine().trim()));
				} catch (ParseException e) {
					System.out.println("Illegal date"); // dialog box?
					return;
				}
				int size = Integer.parseInt(in.readLine());
				ArrayList<OrderItem> items = new ArrayList<OrderItem>(size);
				for (int i = 0; i < size; ++i) {
					OrderItem o = restoreOrderItem(in);
					items.add(o);
				}

				Order order = new Order(cust, billingAddress, shipAddress,
						terms, items, od, sd, cd);

				in.readLine(); // separator
			}
		} catch (IOException ex) {
			System.out.println("Order restore failed. " + ex);
			System.exit(1);
		}
	}

	/**
	 * Write all invoices onto a readable file in a form that can later be read
	 * 
	 * @param out
	 *            the stream to write to
	 */
	private static void dumpInvoices(BufferedWriter out) {
		try {
			out.write("INVOICES");
			out.newLine();
			ArrayList<String> list = sortedKeys(Invoice.keys());
			Iterator<String> it = list.iterator();
			Invoice invoice;
			while (it.hasNext()) {
				invoice = Invoice.get(it.next());
				out.write(invoice.key());
				out.newLine();
				out.write(invoice.order().key());
				out.newLine();
				out.write("" + invoice.size());
				out.newLine();
				Iterator<OrderItem> j = invoice.shipableItems();
				while (j.hasNext()) {
					OrderItem item = j.next();
					dumpOrderItem(item, out);
				}
				out.newLine();
			}
		} catch (IOException ex) {
			System.out.println("Invoice Write Failed. " + ex);
		}
	}

	/**
	 * Retrieve all invoices from the backup file
	 * 
	 * @param in
	 *            the stream to read from
	 */
	private static void restoreInvoices(BufferedReader in) {
		try {
			String num;
			while ((num = in.readLine()) != null && !num.equals("INVENTORY")) {
				Order order = Order.get(in.readLine().trim());

				int size = Integer.parseInt(in.readLine());
				ArrayList<OrderItem> items = new ArrayList<OrderItem>(size);
				for (int i = 0; i < size; ++i) {
					OrderItem o = restoreOrderItem(in);
					items.add(o);
				}

				Invoice invoice = new Invoice(order, items);

				in.readLine(); // separator
			}
		} catch (IOException ex) {
			System.out.println("Order restore failed. " + ex);
			System.exit(1);
		}
	}

	/**
	 * Dump all product records to the backup readable file
	 * 
	 * @param out
	 *            the file to write to
	 */
	private static void dumpProducts(BufferedWriter out) {
		try {
			out.write("PRODUCTS");
			out.newLine();
			//			ArrayList<String> list = sortedKeys(Product.keys()); // get the product keys in key number order
			//			Iterator<String> it = list.iterator();
			//			Product p;
			//			while (it.hasNext())
			//			{
			//				p = Product.get(it.next()); // get the product with this key
			//				out.write(p.key());
			//				out.newLine();
			//				out.write(p.unitPrice());
			//				out.newLine();
			//				out.write("" + p.description());
			//				out.newLine();
			//				out.newLine();// separator between products
			//			}
		} catch (IOException ex) {
			System.out.println("Products Write Failed. " + ex);
		}
	}

	/**
	 * Restore the product records from the readable backup
	 * 
	 * @param in
	 *            the stream to read from
	 */
	private static void restoreProducts(BufferedReader in) {
		try {
			String num;
			while ((num = in.readLine()) != null && !num.equals("ORDERS")) // this line is ignored, the IDnum
			{
				//				String unitPrice = in.readLine();
				//				String description = in.readLine();
				//				Product p = new Product(unitPrice, description); // automatically added

				in.readLine(); // separator line
			}
		} catch (IOException ex) {
			System.out.println("Product Restore Failed. " + ex);
			System.exit(1);
		}
	}

	/**
	 * Dump all inventory records to the backup readable file
	 * 
	 * @param out
	 *            the file to write to
	 */
	private static void dumpInventory(BufferedWriter out) {
		try {
			out.write("INVENTORY");
			out.newLine();
			// more
		} catch (IOException ex) {
			System.out.println("Inventory Write Failed. " + ex);
		}
	}

	/**
	 * Restore the inventory records from the readable backup
	 * 
	 * @param in
	 *            the stream to read from
	 */
	private static void restoreInventory(BufferedReader in) {
		// more. Like the above, except no next section here.
	}

	private static SimpleDateFormat sdf = new SimpleDateFormat(
			"EEE MMM dd yyyy");

	public static void main(final String[] args) throws IOException {// OrderProcessor.open(OrderProcessor.databaseName());
		// System.out.println(keys());

		if (args.length < 2) {
			System.out
					.println("Usage requires parameters 'save filename' OR 'restore filename'");
			return;
		}
		String filename = args[1];
		String command = args[0].toUpperCase();
		if (command.equals("SAVE")) {// Save in text file.
			System.out.println("Saving");
			BufferedWriter out = new BufferedWriter(new FileWriter(filename));
			OrderProcessor.open(OrderProcessor.databaseName());
			dumpCustomers(out);
			dumpProducts(out);
			dumpOrders(out);
			dumpInvoices(out);
			dumpInventory(out);
			out.close();
			System.out.println("Done saving to: " + filename);
		} else if (command.equals("RESTORE")) {// Restore from textfile.
			System.out.println("Restoring.");
			BufferedReader in = new BufferedReader(new FileReader(filename));
			initDatabase(); // Make database empty.
			restoreCustomers(in);
			restoreProducts(in);
			restoreOrders(in);
			restoreInvoices(in);
			restoreInventory(in);
			OrderProcessor.close(OrderProcessor.databaseName());
			System.out.println("Done restoring from: " + filename);
		} else {
			System.out
					.println("Usage requires parameters 'save filename' OR 'restore filename'");
		}

	}
}
