package orderprocessing;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.io.Serializable;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.BufferedWriter;

//import java.io.Writer;

/** This class represents a single order from a single customer.
 * <p>
 *	Additionally, this class maintains a collection of all Orders.
 */
public class Order implements Serializable {
	
	private String key;

	private Customer customer;

	private Address billingAddress;

	private Address shipAddress;

	private ArrayList<OrderItem> orderedItems; // ArrayList of OrderItems

	private CalculationCalendar orderDate;

	private CalculationCalendar requestedShippingDate;

	private CalculationCalendar cancelDate;// if null order is still open

	private BillingTerms terms = BillingTerms.get("NET 10");

	private boolean cancelled = false;

	private boolean validOrder = true;

	private static final int MIN_SHIP_DELAY = 7;

	private static final int MIN_CANCEL_DELAY = 10;

	
	/** Create an Order for a particular customer and a sequence of items ordered
	 * @param customer The customer ordering these items
	 * @param shipAddress The address the order should be shipped to.
	 * @param orderedItems A ArrayList of OrderItems representing the items ordered
	 * @param orderDate The date that this order was placed
	 */
	public Order(Customer customer, Address billingAddress,
			Address shipAddress, ArrayList<OrderItem> orderedItems,
			CalculationCalendar orderDate) {
		this(customer, billingAddress, shipAddress, null, orderedItems,
				orderDate, null, null);
		// uses default ship date
	}

	/** Create an Order for a particular customer and a sequence of items ordered
	 * @param customer The customer ordering these items
	 * @param shipAddress The Shipping Address if different from the customer's default
	 * @param billingTerms The billing terms for order if not the customer's default.
	 * @param orderedItems A ArrayList of OrderItems representing the items ordered
	 * @param orderDate The date that this order was placed
	 * @param requestedShippingDate Requested shipping date by customer
	 * @param cancelDate Requested cancelation date for this order.
	 */
	public Order(Customer customer, Address billingAddress,
			Address shipAddress, BillingTerms billingTerms,
			ArrayList<OrderItem> orderedItems, CalculationCalendar orderDate,
			CalculationCalendar requestedShippingDate,
			CalculationCalendar cancelDate) {
		key = IDGenerator.nextID();
		this.customer = customer;
		this.billingAddress = billingAddress;
		this.shipAddress = shipAddress;
		this.orderedItems = (ArrayList<OrderItem>) orderedItems.clone();
		this.orderDate = orderDate;
		this.terms = billingTerms;

		if (this.terms == null) {
			this.terms = customer.billingTerms();
		}

		CalculationCalendar earlyShipDate = this.orderDate.incrementedBy(
				CalculationCalendar.DATE, MIN_SHIP_DELAY);
		this.requestedShippingDate = earlyShipDate.max(requestedShippingDate);

		CalculationCalendar earlyCancelDate = this.requestedShippingDate
				.incrementedBy(CalculationCalendar.DATE, MIN_CANCEL_DELAY);
		this.cancelDate = earlyCancelDate.max(cancelDate);

		allOrders.put(key, this);
	}

	// accessors
	/** Retrieve the customer for this order
	 *  @return the customer
	 */
	public Customer customer() {
		return customer;
	}

	/** Retrieve the billing address for this order's customer
	 * @return the billing address
	 */
	public Address billingAddress() {
		return billingAddress;
	}

	/** Retrieve the shipping address for this order
	 * @return the ship to address
	 */
	public Address shippingAddress() {
		return shipAddress;
	}

	/** Provide an iteration service for the items ordered on this order
	 * @return an iterator over the ordered items
	 */
	public Iterator<OrderItem> orderedItems() {
		return orderedItems.iterator();
	}

	/** Retrieve the number of order items on this order
	 * @return the number of order items
	 */
	public int size() {
		return orderedItems.size();
	}

	/** Retrieve the date this order was created
	 * @return the order date
	 */
	public CalculationCalendar orderDate() {
		return orderDate;
	}

	/** Retrieve the date the customer would like the order shipped
	 * @return the customer's requested ship date
	 */
	public CalculationCalendar requestedShippingDate() {
		return requestedShippingDate;
	}

	/** Retrieve the last date on which the customer may cancel the order
	 * @return the order's cancelation date
	 */
	public CalculationCalendar cancelDate() {
		return cancelDate;
	}

	/** Return whether this is a valid order or not
	 * @return true if the order is valid
	 */
	public boolean valid() {
		return validOrder;
	}

	/** Retrieve the order number for this order
	 * @return the system generated order number
	 */
	public String key() {
		return key;
	}

	/** Retrieve the billing terms for this order
	 * @return the billing terms
	 */
	public BillingTerms billingTerms() {
		return terms;
	}

	/** Return whether this order has been cancelled or not
	 * @return true if the order was cancelled
	 */
	public boolean isCancelled() {
		return cancelled;
	}

	/** return the extended total dollar value of this order
	 * @return the total value
	 */
	public double total() {
		Iterator<OrderItem> items = orderedItems();
		double total = 0.0;
		while (items.hasNext()) {
			OrderItem ordered = items.next();
			total += ordered.quantity() * ordered.unitPrice();
		}
		return total;
	}

	// mutators
	/** Cancel this order
	 * 
	 */
	public void cancelOrder() {
		cancelled = true;
	}

	/** Assign new billing terms to this order
	 * @param billingTerms the new terms
	 */
	public void billingTerms(BillingTerms billingTerms) {
		terms = billingTerms;
	}

	/** Format all known orders (not just outstanding ones) and print
	 * to a file
	 * @param file the stream to write onto
	 */
	public static void formatAll(BufferedWriter file) // NOTE, NOT in order number order.
	{
		Iterator<Order> orders = orders();
		Order order = null;
		while (orders.hasNext()) {
			try {
				order = orders.next();
				order.format(file);
				file.newLine();
				file.newLine();
			} catch (IOException ex) {
				System.out.println("Failed to write order " + order.key()
						+ "  " + ex);
			}
		}
	}

	/** Format this order in a readable way and write on a stream
	 * @param file the stream to write to
	 * @throws IOException
	 */
	public void format(BufferedWriter file) throws IOException { // TODO format orders for printing
	}

	/** Give this order a new ship address
	 * @param address the new ship address
	 */
	public void shippingAddress(Address address) {
		shipAddress = address;
	}

	// other methods
	/** Display this order on standard output (mostly for debugging)
	 * 
	 */
	public void display() {
		System.out.println("\t\t\t------- Order ------- " + key);
		customer.display();

		System.out.println("\n\n\tQty\tItem No. "
				+ "\tDescription\t\tUnit Cost\n");

		Iterator<OrderItem> items = orderedItems();
		while (items.hasNext()) {
			(items.next()).display();
		}

		terms.display();

		System.out.println("Order    Date: " + orderDate.getTime());
		System.out.println("Customer Shipping information:");
		System.out.println(shippingAddress());
		System.out.println(" Ship Date: " + requestedShippingDate.getTime());
		System.out.println(" Cancel Date: " + cancelDate.getTime());

		System.out.println("---------------------");
	}

	public boolean equals(Object other) {
		if (!(other instanceof Order))
			return false;
		return key.equals(((Order) other).key);
	}

	public int hashcode() {
		return key.hashCode() * 997;
	}

	public String toString() {
		return "Order with id: " + key;
	}

	private static class IDGenerator {
		public static String nextID() {
			return ID_PREFIX + Long.toString(nextID++);
		}

		private static long nextID = 2001;

		private final static String ID_PREFIX = "O";
	}

	// Following is the collection of all Orders.
	private static HashMap<String, Order> allOrders = new HashMap<String, Order>();

	/** Retrieve an order from its system assigned order number
	 * @param key the order number of the desired order
	 * @return the order requested or null if none
	 */
	public static Order get(String key) {
		return allOrders.get(key);
	}

	/** Write out all the orders at the end of a run to the saved database
	 * @param out the Object stream to write to
	 * @throws IOException
	 */
	public static final void close(ObjectOutputStream out) throws IOException {
		out.writeLong(IDGenerator.nextID);
		out.writeObject(allOrders);
	}

	/** Read in all the orders at the beginning of a run from the saved
	 * database
	 * @param in the stream to read from
	 * @throws IOException
	 * @throws ClassNotFoundException
	 */
	public static final void open(ObjectInputStream in) throws IOException,
			ClassNotFoundException {
		IDGenerator.nextID = in.readLong();
		allOrders = (HashMap<String, Order>) in.readObject();
	}

	/** Provide an iteration service over all orders
	 * @return an iterator over the orders. The iterator returns Order objects.
	 */
	public static Iterator<Order> orders() {
		return allOrders.values().iterator();
	}

	/** Provide an iteration service over the order numbers that have 
	 * been created
	 * @return an iteration over the (String) order numbers
	 */
	public static Iterator<String> keys() {
		return allOrders.keySet().iterator();
	}

	/** Show all of the orders on standard output
	 * 
	 */
	public static void dump() {
		System.out.println("\n\n\n Orders \n\n\n");
		System.out.println(allOrders);
	}
}

/* notes shipdate  after order date + 7
 cancel date after ship date + 10
 */