package orderprocessing;

import javax.swing.JButton;
import javax.swing.JComboBox;
import java.awt.Container;
import java.awt.Color;
import java.awt.Component;
import javax.swing.JFrame;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import javax.swing.JLabel;
import javax.swing.JTextArea;
import javax.swing.JTextField;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.awt.Insets;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Calendar;

/**
 *	OrderEntry is the GUI to this application for order entry. The application can
 *    have other front ends for other purposes such as customer maintenance.
 *	Generally it is well designed, with a good event structure. However, it does
 *	incorporate some logic that should be elsewhere, as noted.
 *	It also writes to System.out in places in which it should use dialog boxes instead.
 * 	To do order entry, this should be your "main" class.
 */
public class OrderEntry extends JFrame {
	
	private JTextField customerNumberField = new JTextField(10);

	private JTextField customerNameField = new JTextField(30);

	private JTextArea customerAddress = new JTextArea("", 5, 40); //,
			//TextArea.SCROLLBARS_NONE);

	private JTextArea shippingAddress = new JTextArea("", 5, 40); //,
			//TextArea.SCROLLBARS_NONE);

	private JComboBox billingTerms = new JComboBox();

	private JTextField orderDateField = new JTextField(25);

	private JTextField shipDateField = new JTextField(25);

	private JTextField cancelDateField = new JTextField(25);

	private JTextField itemNumberField = new JTextField(10);

	private JTextField quantityField = new JTextField(10);

	private JTextField priceField = new JTextField(10);

	private JTextField descriptionField = new JTextField(30);

	private JButton accept = new JButton("Accept Item");

	private JButton enter = new JButton("Enter Complete Order");
	
	private JButton save = new JButton("Save Data");

	private Color original = null;

	private ArrayList<String> items = new ArrayList<String>();

	//	private SimpleDateFormat specifiedDateFormat = new SimpleDateFormat("EEE MMM dd, yyyy");
	private SimpleDateFormat specifiedDateFormat = new SimpleDateFormat("MM/dd/yyyy");

	public OrderEntry() {
		super("OrderEntry");
		setBounds(45, 30, 700, 580);
		addWindowListener(new WindowAdapter() // Anonymous inner class
		{
			@Override
			public void windowClosing(WindowEvent e) {
//				OrderProcessor.close(OrderProcessor.databaseName()); // Do this when you want orders saved.
				System.exit(0);
			}
		});
		GridBagLayout layout = new GridBagLayout();
		getContentPane().setLayout(layout);
		final int NONE = GridBagConstraints.NONE;
		final int BOTH = GridBagConstraints.BOTH;
		final int HORIZONTAL = GridBagConstraints.HORIZONTAL;
		add(new JLabel("Order Entry"), 1, 0, 1, 1, 0.0, 1.0, NONE);
		JButton getCustomerInfo = new JButton("Get Customer Information");
		CustomerListener cl = new CustomerListener();
		customerNumberField.addActionListener(cl);
		getCustomerInfo.addActionListener(cl);
		customerNameField.setEnabled(false);
		//customerAddress.setEnabled(false);
		add(new JLabel("Customer Number"), 0, 1, 1, 1, 0.0, 0.0, NONE);
		add(customerNumberField, 1, 1, 1, 1, 0.0, 1.0, HORIZONTAL);
		add(getCustomerInfo, 2, 1, 1, 1, 0.0, 0.0, NONE);
		add(new JLabel("Customer Name"), 0, 3, 1, 1, 0.0, 0.0, NONE);
		add(customerNameField, 1, 3, 4, 1, 0.0, 1.0, HORIZONTAL);
		add(new JLabel("Billing Address"), 0, 4, 1, 1, 0.0, 0.0, NONE);
		add(customerAddress, 1, 4, 4, 3, 0.0, 1.0, BOTH);
		add(new JLabel("Shipping Address"), 0, 8, 1, 1, 0.0, 0.0, NONE);
		add(new JLabel("(Keep this line structure.)"), 0, 9, 1, 1, 0.0, 0.0, NONE);
		add(shippingAddress, 1, 8, 4, 3, 0.0, 1.0, BOTH);

		add(new JLabel("Billing Terms"), 0, 12, 1, 1, 0.0, 0.0, NONE);
		billingTerms.addItem("Default");
		billingTerms.addItem("NET 10");
		billingTerms.addItem("NET 30");
		billingTerms.addItem("EOM 10");
		add(billingTerms, 1, 12, 1, 1, 0.0, 0.0, NONE);
		
		add(new JLabel("Order Date"), 0, 13, 1, 1, 0.0, 0.0, NONE);
		add(new JLabel("Ship Date"), 1, 13, 1, 1, 0.0, 0.0, NONE);
		add(new JLabel("Cancel Date"), 2, 13, 1, 1, 0.0, 0.0, NONE);
		add(orderDateField, 0, 14, 1, 1, 0.0, 0.0, HORIZONTAL);
		add(shipDateField, 1, 14, 1, 1, 0.0, 0.0, HORIZONTAL);
		add(cancelDateField, 2, 14, 1, 1, 0.0, 0.0, HORIZONTAL);
		add(new JLabel("Item Number (numeric)"), 0, 16, 1, 1, 0.0, 0.0, NONE);
		add(itemNumberField, 1, 16, 1, 1, 0.0, 0.0, HORIZONTAL);
		add(new JLabel("Quantity"), 0, 17, 1, 1, 0.0, 0.0, NONE);
		add(quantityField, 1, 17, 1, 1, 0.0, 0.0, HORIZONTAL);
		add(new JLabel("Unit Price"), 0, 18, 1, 1, 0.0, 0.0, NONE);
		add(priceField, 1, 18, 1, 1, 0.0, 0.0, HORIZONTAL);
		add(new JLabel("Description"), 0, 19, 1, 1, 0.0, 0.0, NONE);
		add(descriptionField, 1, 19, 1, 1, 0.0, 0.0, HORIZONTAL);
		add(accept, 0, 20, 1, 1, 0.0, 0.0, NONE);
		add(enter, 1, 20, 1, 1, 0.0, 0.0, NONE);
		add(save, 2, 20, 1, 1, 0.0, 0.0, NONE);
		enter.setVisible(false);

		accept.addActionListener(new AcceptListener());
		enter.addActionListener(new EnterListener());
		save.addActionListener(new SaveListener());
		setBackground(new Color(175, 175, 255));
		setVisible(true);
		original = enter.getBackground();
	}
	
	

	/** Convenience method to aid adding a component to a grid bag layout
	 * @param arg The component to be added
	 * @param x the x location (column) in the layout for the upper edge
	 * @param y the row in the layout for the left edge
	 * @param h the number of cells to use horizontally
	 * @param v the number of cells to used vertically
	 * @param wx how to balance horizontally if not all cells are filled
	 * @param wy how to balance vertically if not all cells are filled
	 */
	private void add(Component arg, int x, int y, int h, int v, double wx,
			double wy, int fill) {
		Container content = getContentPane();
		GridBagLayout gbl = (GridBagLayout) content. getLayout();
		GridBagConstraints c = new GridBagConstraints();
		c.fill = GridBagConstraints.NONE;
		c.gridx = x;
		c.gridy = y;
		c.gridwidth = h;
		c.gridheight = v;
		c.weightx = wx;
		c.weighty = wy;
		c.fill = fill;
		c.ipadx = 5;
		c.ipady = 5;
		c.insets = new Insets(1,1,1,1);
		gbl.setConstraints(arg, c);
		content. add(arg);
		
	}

	/**
	 * Listener for the customer JButton and customer number text field
	 *
	 */
	private class CustomerListener implements ActionListener {
		public void actionPerformed(ActionEvent e) {
			String customerNumber = customerNumberField.getText().trim();
			if (customerNumber.equals("")) {
				customerNameField.setText("Enter a customer number above.");
				customerNumberField.requestFocus();
				return;
			}
			customerNumber = customerNumber.toUpperCase();
			customerNumberField.setText(customerNumber);
			Customer customer = Customer.get(customerNumber);
			if (customer == null) {
				customerNameField.setText("No such customer.");
				customerAddress.setText("");
				shippingAddress.setText("");
				billingTerms.setSelectedItem("Default");
				customerNumberField.selectAll();
				customerNumberField.requestFocus();
				return;
			}
			customerNameField.setText(customer.name());
			customerAddress.setText("" + customer.billingAddress());
			shippingAddress.setText("" + customer.shippingAddress());

			billingTerms.setSelectedItem(customer.billingTerms().code()); // is equivalent to the above.

			CalculationCalendar orderDate = new CalculationCalendar();
			orderDateField.setText(specifiedDateFormat.format(orderDate.getTime()));
			CalculationCalendar shipDate = orderDate.incrementedBy(Calendar.DAY_OF_MONTH, 7);
			CalculationCalendar cancelDate = shipDate
					.incrementedBy(Calendar.DAY_OF_MONTH, 10);
			shipDateField.setText(specifiedDateFormat.format(shipDate.getTime()));
			cancelDateField.setText(specifiedDateFormat.format(cancelDate.getTime()));
			itemNumberField.requestFocus();

			items.clear(); // make sure any earlier items are removed.
			enter.setBackground(original);
			enter.setText("Enter Complete Order");
		}
	}

	/**
	 * Listener for the accept order item JButton
	 *
	 */
	private class AcceptListener implements ActionListener {
		public void actionPerformed(ActionEvent evt) { /* Note that only blank field editing has been done. It needs more.
		 The logic of the editing should be put with the appropriate objects
		 in an "edit" method and not here.
		 */
			String item = "";
			String temp = itemNumberField.getText().trim();
			if (temp.equals("")) {
				itemNumberField.setText("Illegal");
				itemNumberField.selectAll();
				itemNumberField.requestFocus();
				return;
			}
			item += temp + "~";
			temp = quantityField.getText().trim();
			if (temp.equals("")) {
				quantityField.setText("Illegal");
				quantityField.selectAll();
				quantityField.requestFocus();
				return;
			}
			item += temp + "~";
			temp = priceField.getText().trim();
			if (temp.equals("")) {
				priceField.setText("Illegal");
				priceField.selectAll();
				priceField.requestFocus();
				return;
			}
			item += temp + "~";
			temp = descriptionField.getText().trim();
			if (temp.equals("")) {
				descriptionField.setText("Illegal");
				descriptionField.selectAll();
				descriptionField.requestFocus();
				return;
			}
			item += temp;
			//System.out.println(item); // debug output
			items.add(item);
			itemNumberField.setText("");
			quantityField.setText("");
			priceField.setText("");
			descriptionField.setText("");
			item = "";
			itemNumberField.requestFocus();
			enter.setBackground(Color.yellow);
			enter.setText("Save Unsaved Order ("+ items.size()+" items)");
			enter.setVisible(0 != items.size());
		}
	}

	/**
	 * Listener for the enter order JButton
	 *
	 */
	private class EnterListener implements ActionListener {
		public void actionPerformed(ActionEvent evt) {
			CalculationCalendar orderDate = new CalculationCalendar();
			CalculationCalendar shipDate = new CalculationCalendar();
			CalculationCalendar cancelDate = new CalculationCalendar();
			try {
				orderDate.setTime(specifiedDateFormat.parse(orderDateField.getText()));
				shipDate.setTime(specifiedDateFormat.parse(shipDateField.getText()));
				cancelDate.setTime(specifiedDateFormat.parse(cancelDateField.getText()));
			} catch (ParseException e) {
				System.out.println("Illegal date"); // dialog box?
				return;
			}

			if (items.size() == 0) {
				System.out.println("There are no items ordered."); // dialog box?
				return;
			}

			String terms = (String) billingTerms.getSelectedItem();

			String billAddress = customerAddress.getText().replace('\n', '~');

			// Encode the ship address as required by its parser.
			// We should at least count the fields.
			String shipAddress = shippingAddress.getText().replace('\n', '~');
			// System.out.println(shipAddress); // debug output

			// System.out.println(items); // debug output

			String customer = customerNumberField.getText().trim();

			try { // create the order object.
				Order orderProcessor = OrderProcessor.orderProcessor.newOrder(customer, terms,
						billAddress, shipAddress, items, orderDate, shipDate, cancelDate);
//				if (orderProcessor == null)// if customer is illegal
//				{
//					System.out.println("Failed order--No such customer."); // dialogs?
//					enter.setVisible(false);
//					items.clear();
//					return;
//				}
				// Not saving in database. Verify creation:
				orderProcessor.display();
				// Testing only:
				OrderProcessor.orderProcessor.generateInvoice(orderProcessor).display();
			} catch (Exception ex) {
				System.out.println("Order creation failed.  " + ex);// dialogs?
			}
			items.clear();
			customerNumberField.selectAll();
			customerNumberField.requestFocus();
			enter.setBackground(original);
			enter.setText("Enter Complete Order");
			enter.setVisible(false);
		}
	}
	
	private class SaveListener implements ActionListener{
		public void actionPerformed(ActionEvent e){
			OrderProcessor.close(OrderProcessor.databaseName()); // Do this when you want orders saved.	
			System.out.println("Saved data.");
		}
	}

	/** Create and display the order entry gui. Opens the database.
	 * @param args ignored
	 */
	public static void main(final String[] args) {
		OrderProcessor.open(OrderProcessor.databaseName());
		new OrderEntry();
	}

}
