package orderprocessing;

import java.util.StringTokenizer;

/*
 History
 date		description
 -----		-----------
 11/12/99	Add item description
 */

import java.io.Serializable;
import java.text.NumberFormat;

/** This class contains information about a single item on a customer order.
 * Note that this class implements the Immutable Object design pattern.
 * <p> These are used for both orders and invoices.
 */

public class OrderItem implements Serializable, Cloneable {
	
	private int quantity;

	private long itemNumber;

	private double unitPrice;

	private String description;

	/** Create an order item for a given quantity of a certain item
	 * @param quantity The quantity ordered.
	 * @param itemNumber The item desired.
	 * @param unitPrice The price per unit for this item.
	 * @param description - Item description
	 */
	public OrderItem(int quantity, // TODO Should we guarantee >0? >=0?
			long itemNumber, //  TODO Should this be a long or a code like I3001?
			double unitPrice, // TODO What validation is needed for this?
			String description) {
		this.quantity = quantity;
		this.itemNumber = itemNumber;
		this.unitPrice = unitPrice;
		this.description = description;
	}

	/** Retrieve the quantity for this order item
	 * @return the quantity ordered
	 */
	public int quantity() {
		return quantity;
	}

	/** Retrieve the item number for this order item
	 * @return the item number TODO should probably be a code
	 */
	public long itemNumber() {
		return itemNumber;
	}

	/** Retrieve the unit price of this item
	 * @return the price TODO should we worry about precision?
	 */
	public double unitPrice() {
		return unitPrice;
	}

	/** Retrieve the extended price (unit price times quantity) for this 
	 * order item
	 * @return the extended price
	 */
	public double extendedPrice() {
		return unitPrice * quantity;
	}

	/** Retrieve the description of the item in this order item
	 * @return the description of the item
	 */
	public String description() {
		return description;
	}

	public boolean equals(Object o) {
		if (!(o instanceof OrderItem)) {
			return false;
		}
		OrderItem oi = (OrderItem) o;
		return quantity == oi.quantity && itemNumber == oi.itemNumber
				&& unitPrice == oi.unitPrice;
	}

	public int hashCode() {
		return (int) ((quantity + 1) * unitPrice * itemNumber);
	}

	// Guarantee that two "equals" OrderItems have the same hash code.

	/** Display this order item on standard out
	 * 
	 */
	public void display() {
		System.out.println("\t" + quantity + "\t" + itemNumber + "\t\t"
				+ description + "\t\t"
				+ NumberFormat.getInstance().format(unitPrice));

	}

	/** Display the order item along with its extended price on standard
	 *  out.
	 */
	public void displayExtended() {
		System.out.println("\t" + quantity + "\t" + itemNumber + "\t\t"
				+ description + "\t\t"
				+ NumberFormat.getInstance().format(unitPrice * quantity));
	}

	public Object clone() {
		return new OrderItem(quantity, itemNumber, unitPrice, description);
	}

	public String toString() {
		return quantity + " " + itemNumber + " " + unitPrice;
	}

	/** Generate a new order item from a correctly formatted string. Caller
	 * is responsible for the correctness. 
	 * @param orderItem the string describing the order item
	 * @param sep the separator character(s) used in the description string
	 * @return a new Order item
	 */
	public static OrderItem generateOrderItem(String orderItem, String sep) {
		StringTokenizer tok = new StringTokenizer(orderItem, sep);
		//System.out.println("--> " + orderItem); // Debug code.
		long itemNumber = Long.parseLong(tok.nextToken().trim());
		int quantity = Integer.parseInt(tok.nextToken().trim());
		double unitPrice = 0.0;
		try {
			unitPrice = Double.valueOf(tok.nextToken().trim()).doubleValue();
		} catch (NumberFormatException e) {
			System.out.println("Illegal value for price.");
		}
		String description = tok.nextToken().trim();
		return new OrderItem(quantity, itemNumber, unitPrice, description);
	}

}
