package orderprocessing;

import java.util.ArrayList;
import java.util.Iterator;
//import java.util.StringTokenizer;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.FileInputStream;
import java.io.FileOutputStream;

/**
 * This class accepts and fulfills orders from customers. It can create an order
 * from appropriate information and it can create an invoice from an order. It
 * also maintains the logic for opening and closing the object database as the
 * program starts and quits. It also does some writing to System.out that would
 * better be handled by dialogs.
 */

public class OrderProcessor {
	
	private static String databaseName = "test";

	private static String databaseExtension = ".dat";

	private static int version = 0; // for autobackup

	private InventoryManager inventory;

	private static ObjectInputStream in = null;

	/**
	 * Create a new OrderProcessor for a given InventoryManager
	 * 
	 * @param inventory
	 *            The InventoryManager that will resolve orders. Note that this
	 *            class implements the Singleton Object (Unique Object) design
	 *            pattern.
	 */
	private OrderProcessor(InventoryManager inventory) {
		this.inventory = inventory;
	}

	/**
	 * Generate an order object from string data sent to it. This method acts
	 * like a factory for Orders.
	 * 
	 * @param customer
	 *            String with customer number (C5001, C5001,...)
	 * @param shipping
	 *            The shipping address normally associated with this customer
	 *            (will change)
	 * @param items
	 *            An ArrayList of strings each of which is a quantity,
	 *            itemnumber, price, and. description. There is a tide ~
	 *            separation between the fields.
	 * @param orderDate
	 *            The date of this order.
	 * @param requestedShipDate
	 *            the date shipment is requested (Order date + 7days).
	 * @param cancelDate
	 *            the date the order should be cancelled if not yet shipped.
	 * @return an Order for these items or null if the customer is invalid. Note
	 *         that this is a Factory Method (pattern) for Order objects
	 */
	public Order newOrder(String customer, String terms, String billing,
			String shipping, ArrayList items, CalculationCalendar orderDate,
			CalculationCalendar requestedShipDate,
			CalculationCalendar cancelDate) {
		Customer c = Customer.get(customer);
		if (c == null) {
			return null;
		}
		Iterator e = items.iterator();
		ArrayList orderItems = new ArrayList();
		while (e.hasNext()) {
			OrderItem result = OrderItem.generateOrderItem((String) e.next(), "~");
			orderItems.add(result);
		}
		Address billingAddress;
		if (billing.trim().equals("")
				|| shipping.toUpperCase().indexOf("DEFAULT") >= 0) {
			billingAddress = c.billingAddress();
		} else {
			billingAddress = new Address(billing, "~");
		}

		Address shippingAddress;
		if (shipping.trim().equals("")
				|| shipping.toUpperCase().indexOf("DEFAULT") >= 0) {
			shippingAddress = billingAddress;
		} else {
			shippingAddress = new Address(shipping, "~");
		}
		BillingTerms billingTerms = BillingTerms.get("NET 10");
		terms = terms.toUpperCase();
		if (terms.indexOf("DEFAULT") >= 0) {
			billingTerms = c.billingTerms();
		} else {
			if (terms.indexOf("NET 10") >= 0) {
				billingTerms = BillingTerms.get("NET 10");
			} else if (terms.indexOf("NET 30") >= 0) {
				billingTerms = BillingTerms.get("NET 30");
			} else if (terms.indexOf("EOM 10") >= 0) {
				billingTerms = BillingTerms.get("EOM 10");
			}
		}
		return new Order(c, billingAddress, shippingAddress, billingTerms,
				orderItems, orderDate, requestedShipDate, cancelDate);
	}

	/**
	 * Generate an invoice from an order.
	 * 
	 * @param order
	 *            An order previously placed by a customer.
	 * @return An Invoice for items that can be shipped.
	 */
	public Invoice generateInvoice(Order order) {
		Iterator e = order.orderedItems();
		ArrayList invoicedItems = new ArrayList();
		while (e.hasNext()) {
			OrderItem trial = (OrderItem) e.next();
			OrderItem result = inventory.available(trial);
			if (result.quantity() > 0) {
				invoicedItems.add(result);
			} else {
				System.out.println(trial + " not available"); // Fix this.// (Exceptions/dialogs?)
			}

		}
		return new Invoice(order, invoicedItems);
	}

	public static OrderProcessor orderProcessor = new OrderProcessor(
			InventoryManager.standardInventoryManager);

	/**
	 * close the database at the end of processing. Write it to the saved
	 * version. Must be compatible with open.
	 * 
	 * @param filename
	 *            the name of the file to write onto
	 */
	public static synchronized void close(String filename) {
		try {
			in.close(); // just in case.
		} catch (IOException e) {
		} catch (NullPointerException npe) {
		}

		try {
			ObjectOutputStream out = new ObjectOutputStream(
					new FileOutputStream(filename));
			// out.writeInt(version); // for autobackup
			Customer.close(out);
			Order.close(out);
			Invoice.close(out);
			InventoryManager.close(out);
			out.close();
		} catch (IOException e) {
			System.out.println("Failed writing database. " + e);
		}
	}

	/**
	 * Open the database at the beginning of processing by reading it into
	 * memory from the saved version. Must be compatible with close.
	 * 
	 * @param filename
	 *            the name of the file to read from
	 */
	public static void open(String filename) {
		try {
			in = new ObjectInputStream(new FileInputStream(filename));
			// version = in.readInt()+1; // for autobackup
			Customer.open(in);
			Order.open(in);
			Invoice.open(in);
			InventoryManager.open(in);
			// This stream is held open until the program halts or until it
			// calls close.
			// close(databaseName+version+databaseExtension); // for autobackup
		} catch (IOException e) {
			System.out.println("Failed restoring database. " + e);
		} catch (ClassNotFoundException ce) {
			System.out.println("Class inconsistency. " + ce);
		}
	}


	/**
	 * The name of the database file
	 * 
	 * @return the name of the file
	 */
	public static String databaseName() {
		return databaseName + databaseExtension;
	}

	/**
	 * Close the database using the default name If the database was not opened
	 * previously, this will lose all of the saved records.
	 */
	private static void closeDatabase() {
		close(databaseName);
	}

	/**
	 * This method is used only to initialize the database to empty after a
	 * catastrophic failure in which the data must all be rebuild from external
	 * data.It will leave you with an empty database.
	 */
	public static void main(String[] args) {
		closeDatabase();
	}
}

/*
 * Note that money is handled naively here. To do a better job with money, see
 * Fowler's Analysis Patterns. Another naive solution would use a long (integer)
 * representing pennies (or possibly tenths of pennies...).
 */

/*
 * Note that writing an "error" message to standard output is NEVER adequate
 * error handling in a program of this sort for other than run failures. (See
 * generateInvoice.)
 */

